
#include <gtest/gtest.h>
#include <iostream>
#include <string>
#include <variant>

using VariantType = std::variant<int, std::string, bool>;

struct FunctorImpl {
  auto operator()(const int& v) { std::cout << v << '\n'; }
  auto operator()(const std::string& v) { std::cout << v << '\n'; }
  auto operator()(const bool& v) { std::cout << v << '\n'; }
};

auto visit_impl(FunctorImpl ifunctor, const VariantType& ivariant) {
  if (std::holds_alternative<int>(ivariant)) {
    return ifunctor(std::get<int>(ivariant));
  } else if (std::holds_alternative<std::string>(ivariant)) {
    return ifunctor(std::get<std::string>(ivariant));
  } else if (std::holds_alternative<bool>(ivariant)) {
    return ifunctor(std::get<bool>(ivariant));
  }
}

TEST(Variant, Assignment) {
  auto v = VariantType{};                      // Wariant jest pusty
  ASSERT_TRUE(std::holds_alternative<int>(v)); // Prawda, int jest pierwszy z wariantów.
  v = 7;
  ASSERT_TRUE(std::holds_alternative<int>(v)); // Prawda
  v = std::string{"Anne"};
  ASSERT_FALSE(std::holds_alternative<int>(v)); // Fałsz, wartość typu int została zastąpiona.
  v = false;
  ASSERT_TRUE(std::holds_alternative<bool>(v)); // Prawda, v jest teraz typu bool.
}

TEST(Variant, SizeOf) {

  std::cout << "VariantType: " << sizeof(VariantType) << '\n';
  std::cout << "std::string: " << sizeof(std::string) << '\n';
  std::cout << "std::size_t: " << sizeof(std::size_t) << '\n';
  // VariantType prawdopodobnie ma wielkość sizeof(string) + sizeof(size_t)
}

// Bezpieczeństwo ze względu na wyjątki
//
namespace {
struct Widget {
  explicit Widget(int) { // Konstruktor zgłaszający błędy
    throw std::exception{};
  }
};
} // namespace

TEST(Variant, ExceptionSafety) {
  auto var = std::variant<double, Widget>{1.0};
  try {
    var.emplace<1>(42); // Prób autworzenia instancji
  } catch (...) {
    std::cout << "przechwycono wyjątek\n";
    if (var.valueless_by_exception()) {
      std::cout << "bez wartości\n";
    } else {
      std::cout << std::get<0>(var) << '\n';
    }
  }
}

//
// Używanie wariantów
//

TEST(Variant, StdVisit) {
  auto var = std::variant<int, bool, float>{};
  std::visit([](auto&& val) { std::cout << val; }, var);
  std::cout << std::endl;
}

template <class... Lambdas> struct Overloaded : Lambdas... {
  using Lambdas::operator()...;
};

// Dodawanie bezpośredniej wskazówki dotyczącej wnioskowania typów
// W C++20 ten wiersz nie jest konieczny, ale dodałem go, aby kod był kompatybilny ze starszymi wersjami
template <class... Lambdas> Overloaded(Lambdas...) -> Overloaded<Lambdas...>;

TEST(Variant, Overloaded) {

  auto overloaded_lambdas =
      Overloaded{[](int v) { std::cout << "Int: " << v; },
                 [](bool v) { std::cout << "Bool: " << v; },
                 [](float v) { std::cout << "Float: " << v; }};

  overloaded_lambdas(30031);    // Wyświetla "Int: 30031"
  overloaded_lambdas(2.71828f); // Wyświetla "Float: 2.71828"
}

TEST(Variant, VisitOverloaded) {
  std::variant<int, bool, float> v = 42;

  std::visit(Overloaded{[](int v) { std::cout << "Int: " << v; },
                        [](bool v) { std::cout << "Bool: " << v; },
                        [](float v) { std::cout << "Float: " << v; }},
             v);
  // Wyświetla: "Int: 42"
}