#include <cassert>
#include <gtest/gtest.h>

namespace {

struct Number {

  // Zakładam, że ta klasa może zgłaszać wyjątki przy konstrukcji i przypisaniach
  // Number(const Number& other) {
  //     ...
  //     throw ...
  // }
  // auto& operator=(const Number& other) {
  //     ...
  //     throw ...
  // }
  // ...

  // Stosowanie powtarzalnych porównań (C++20)
  // auto operator<=>(const Number&) const = default; // [ Wymaga C++20 ]
  // Rozwiązanie rezerwowa, jeśli operator<=> nie jest obsługiwany.
  bool operator==(const Number& rhs) const { return i_ == rhs.i_; }
  bool operator!=(const Number& rhs) const { return !(*this == rhs); }
  int i_{};
};

class Widget {
public:
  Widget(const Number& x, const Number& y) : x_{x}, y_{y} {
    assert(is_valid());
  }
  void update(const Number& a, const Number& b);
  bool is_valid() const { // Niezmiennik klasy
    return x_ != y_;
  }

private:
  Number x_{};
  Number y_{};
};

// Implementacja bezpieczna ze względu na wyjątki
// z użyciem idiomu "kopiuj i przestaw"
void Widget::update(const Number& x, const Number& y) {
  assert(x != y); // Warunek wstępny
  assert(is_valid());
  // Wykonywanie operacji, które mogą zgłaszać wyjątki,
  // bez modyfikowania stanu tego obiektu Widget
  auto x_tmp = x; // Może zgłaszać wyjątek, ale obiekt Widget pozostaje poprawny
  auto y_tmp = y; // Może zgłaszać wyjątek, ale obiekt Widget pozostaje poprawny
  // Wyjątki nie zostały zgłoszone, można 
  // bezpiecznie zmienić stan za pomocą funkcji
  // niezgłaszającej wyjątków
  std::swap(x_tmp, x_);
  std::swap(y_tmp, y_);
  assert(is_valid()); // Warunek końcowy
}

TEST(ExceptionSafety, CopyAndSwap) {
  auto w = Widget{{1}, {2}};
  w.update({2}, {3});
  ASSERT_TRUE(w.is_valid());
}

} // namespace