#include "stm32f4xx.h"

#define    GPIOBEN            (1U<<1)
#define    I2C1EN             (1U<<21)

#define    I2C_100KHZ         80
#define    SD_MODE_MAX_RISE_TIME    17
#define    CR1_PE             (1U<<0)
#define    SR2_BUSY           (1U<<1)
#define    CR1_START          (1U<<8)
#define    SR1_SB             (1U<<0)
#define    SR1_ADDR           (1U<<1)
#define    SR1_TXE            (1U<<7)
#define    CR1_ACK            (1U<<10)
#define    CR1_STOP           (1U<<9)
#define    SR1_RXNE           (1U<<6)
#define    SR1_BTF            (1U<<2)

/*
 * PB8 ---SCL
 * PB9 ----SDA
 * */
void i2c1_init(void)
{
    /* Włączanie dostępu do zegara dla GPIOB */
    RCC->AHB1ENR |=GPIOBEN;

    /* Ustawianie trybu alternatywnego dla pinów PB8 i PB9 */
    GPIOB->MODER &=~(1U<<16);
    GPIOB->MODER |=(1U<<17);

    GPIOB->MODER &=~(1U<<18);
    GPIOB->MODER |=(1U<<19);

    /* Ustawianie PB8 i PB9 jako wyjść z otwartym drenem */
    GPIOB->OTYPER |=(1U<<8);
    GPIOB->OTYPER |=(1U<<9);

    /* Włączanie rezystorów podciągających dla PB8 i PB9 */
    GPIOB->PUPDR |=(1U<<16);
    GPIOB->PUPDR &=~(1U<<17);

    GPIOB->PUPDR |=(1U<<18);
    GPIOB->PUPDR &=~(1U<<19);

    /* Ustawianie alternatywnej funkcji I2C (AF4) dla PB8 i PB9 */
    GPIOB->AFR[1] &=~(1U<<0);
    GPIOB->AFR[1] &=~(1U<<1);
    GPIOB->AFR[1] |=(1U<<2);
    GPIOB->AFR[1] &=~(1U<<3);

    GPIOB->AFR[1] &=~(1U<<4);
    GPIOB->AFR[1] &=~(1U<<5);
    GPIOB->AFR[1] |=(1U<<6);
    GPIOB->AFR[1] &=~(1U<<7);

    /* Włączanie dostępu do zegara dla I2C1 */
    RCC->APB1ENR |= I2C1EN;

    /* Wejście w tryb resetu */
    I2C1->CR1 |= (1U<<15);

    /* Wyjście z trybu resetu */
    I2C1->CR1 &=~(1U<<15);

    /* Ustawianie częstotliwości zegara peryferyjnego */
    I2C1->CR2 = (1U<<4);   // 16 MHz

    /* Ustawianie standardowego trybu I2C, zegar 100 kHz */
    I2C1->CCR = I2C_100KHZ;

    /* Ustawianie czasu narastania */
    I2C1->TRISE = SD_MODE_MAX_RISE_TIME;

    /* Włączanie modułu I2C1 */
    I2C1->CR1 |= CR1_PE;

}

void i2c1_byte_read(char saddr, char maddr, char* data) {

    volatile int tmp;

    /* Czekanie, aż magistrala będzie wolna */
    while (I2C1->SR2 & (SR2_BUSY)){}

    /* Generowanie sygnału START */
    I2C1->CR1 |= CR1_START;

    /* Czekanie, aż zostanie ustawiona flaga START */
    while (!(I2C1->SR1 & (SR1_SB))){}

    /* Wysyłanie adresu urządzenia podrzędnego + bitu zapisu */
    I2C1->DR = saddr << 1;

    /* Czekanie, aż zostanie ustawiona flaga adresu */
    while (!(I2C1->SR1 & (SR1_ADDR))){}

    /* Zerowanie flagi adresu */
    tmp = I2C1->SR2;

    /* Wysyłanie adresu pamięci */
    I2C1->DR = maddr;

    /* Czekanie, aż nadajnik będzie pusty */
    while (!(I2C1->SR1 & SR1_TXE)){}

    /* Generowanie ponownego warunku START */
    I2C1->CR1 |= CR1_START;

    /* Czekanie, aż zostanie ustawiona flaga START */
    while (!(I2C1->SR1 & SR1_SB)){}

    /* Wysyłanie adresu urządzenia podrzędnego + bitu odczytu */
    I2C1->DR = saddr << 1 | 1;

    /* Czekanie, aż zostanie ustawiona flaga adresu */
    while (!(I2C1->SR1 & (SR1_ADDR))){}

    /* Wyłączanie potwierdzenia */
    I2C1->CR1 &= ~CR1_ACK;

    /* Zerowanie flagi adresu */
    tmp = I2C1->SR2;

    /* Generowaniu warunku STOP po otrzymaniu danych */
    I2C1->CR1 |= CR1_STOP;

    /* Czekanie, aż zostanie ustawiona flaga RXNE */
    while (!(I2C1->SR1 & SR1_RXNE)){}

    /* Odczytywanie danych z rejestru DR */
    *data++ = I2C1->DR;
}

void i2c1_burst_read(char saddr, char maddr, int n, char* data)
{
	volatile int tmp;

    /* Czekanie, aż magistrala będzie wolna */
    while (I2C1->SR2 & (SR2_BUSY)){}

    /* Generowanie warunku startu */
    I2C1->CR1 |= CR1_START;

    /* Czekanie na ustawienie flagi startu */
    while (!(I2C1->SR1 & SR1_SB)){}

    /* Wysyłanie adresu urządzenia podrzędnego + bitu zapisu */
    I2C1->DR = saddr << 1;

    /* Czekanie na ustawienie flagi adresu */
    while (!(I2C1->SR1 & SR1_ADDR)){}

    /* Zerowanie flagi adresu */
    tmp = I2C1->SR2;

    /* Czekanie, aż nadajnik będzie pusty */
    while (!(I2C1->SR1 & SR1_TXE)){}

    /* Wysyłanie adresu pamięci */
    I2C1->DR = maddr;

    /* Czekanie, aż nadajnik będzie pusty */
    while (!(I2C1->SR1 & SR1_TXE)){}

    /* Generowanie ponownego startu */
    I2C1->CR1 |= CR1_START;

    /* Czekanie na ustawienie flagi startu */
    while (!(I2C1->SR1 & SR1_SB)){}

    /* Wysyłanie adresu urządzenia podrzędnego + bitu odczytu */
    I2C1->DR = saddr << 1 | 1;

    /* Czekanie na ustawienie flagi adresu */
    while (!(I2C1->SR1 & (SR1_ADDR))){}

    /* Zerowanie flagi adresu */
    tmp = I2C1->SR2;

    /* Włączanie potwierdzenia */
    I2C1->CR1 |=  CR1_ACK;

    while(n > 0U)
    {
        /* Jeśli został jeden bajt */
        if(n == 1U)
        {
            /* Wyłączanie potwierdzenia */
            I2C1->CR1 &= ~CR1_ACK;

            /* Generowanie warunku stopu */
            I2C1->CR1 |= CR1_STOP;

            /* Czekanie na ustawienie flagi RXNE */
            while (!(I2C1->SR1 & SR1_RXNE)){}


            /* Odczytywanie danych z rejestru DR */
            *data++ = I2C1->DR;
            break;
        }
        else
        {
            /* Czekanie na ustawienie flagi RXNE */
            while (!(I2C1->SR1 & SR1_RXNE)){}

            /* Odczytywanie danych z rejestru DR */
            (*data++) = I2C1->DR;

            n--;
        }
    }
}

void i2c1_burst_write(char saddr, char maddr, int n, char* data) {

    volatile int tmp;

    /* Czekanie, aż magistrala będzie wolna */
    while (I2C1->SR2 & (SR2_BUSY)){}

    /* Generowanie sygnału startu */
    I2C1->CR1 |= CR1_START;

    /* Czekanie na ustawienie flagi startu */
    while (!(I2C1->SR1 & (SR1_SB))){}

    /* Wysyłanie adresu urządzenia podrzędnego */
    I2C1->DR = saddr << 1;

    /* Czekanie na ustawienie flagi adresu */
    while (!(I2C1->SR1 & (SR1_ADDR))){}

    /* Zerowanie flagi adresu */
    tmp = I2C1->SR2;

    /* Czekanie, aż rejestr danych będzie pusty */
    while (!(I2C1->SR1 & (SR1_TXE))){}

    /* Wysyłanie adresu pamięci */
    I2C1->DR = maddr;

    for (int i = 0; i < n; i++) {
        /* Czekanie, aż rejestr danych będzie pusty */
        while (!(I2C1->SR1 & (SR1_TXE))){}

        /* Wysyłanie danych */
        I2C1->DR = *data++;
    }

    /* Czekanie na zakończenie transferu */
    while (!(I2C1->SR1 & (SR1_BTF))){}

    /* Generowanie sygnału stopu */
    I2C1->CR1 |= CR1_STOP;
}

