/*
  server.c

  To compile:

  cc -c -I/usr/local/SSLeay-0.6.4/include server.c
  cc server.o -L/usr/local/SSLeay-0.6.4/lib -lssl -lcrypto -lsocket -lnsl -o server

This program implements a simple server which accepts TCP/IP connections, starts SSL on the connection, waits for some data (which it prints), sends some data back to the client, then waits for more data.  When the connection is closed by the client, it continues to wait for a new connection.

There are no arguments.  When the server starts, it will tell you what port it is waiting on.  This information is used to start the client.

You will need to supply a certificate for a CA, the server's certificate, and the server's private key.  These are used in CAfile, SERVER_CERT, and SERVER_KEY respectively below.

*/

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stdio.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include "buffer.h"
#include "crypto.h"
#include "../e_os.h"
#include "x509.h"
#include "ssl.h"
#include "err.h"

#define CAfile "demoCA/cacert.pem"
#define CApath NULL
#define SERVER_CERT "./server_cert.pem"
#define SERVER_KEY "./server_key.pem"

main()
{
	int sock;		/* The TCP/IP socket */
	int length;
	struct sockaddr_in server;
	int fd;
	char buf[1024];
	int rval;
	SSL_CTX *s_ctx=NULL;	/* The Server's context */
	SSL *s_ssl=NULL;	/* The Server's SSL connection */
	
	SSL_load_error_strings();

	/* Create a new context.  This holds information pertinent to the
	 * client's SSL side of the connection.
	 */
	s_ctx=SSL_CTX_new();

	if (s_ctx == NULL)
	{
		printf("SSL_CTX_new() failed\n");
	}

	/* Tell SSL where the server's public certificate is */
	if (SSL_CTX_use_certificate_file(s_ctx,SERVER_CERT,
			SSL_FILETYPE_PEM) == 0)
	{
		fprintf(stderr, "SSL_CTX_use_certificate_file:");
		ERR_print_errors_fp(stderr);
		exit(1);
	}

	/* Tell SSL where the server's private key is */
	if (SSL_CTX_use_RSAPrivateKey_file(s_ctx,SERVER_KEY,
			SSL_FILETYPE_PEM) == 0)
	{
		fprintf(stderr, "SSL_CTX_use_RSAPrivateKey_file:");
		ERR_print_errors_fp(stderr);
		exit(1);
	}

	/* Tell SSL where the Certificate Authority files are located */
	if ((!SSL_load_verify_locations(s_ctx,CAfile,CApath)) ||
	    (!SSL_set_default_verify_paths(s_ctx)))
	{
		fprintf(stderr,"SSL_load_verify_locations\n");
		ERR_print_errors_fp(stderr);
		exit(1);
	}

	/* Now we create a socket and wait for a basic TCP/IP connection */
	if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0)
	{
		perror("socket");
		exit(1);
	}

	server.sin_family = AF_INET;
	server.sin_addr.s_addr = INADDR_ANY;
	server.sin_port = 0;
	
	if (bind(sock, (struct sockaddr *)&server, sizeof(server)) == -1)
	{
		perror("bind");
		exit(1);
	}

	length = sizeof(server);

	if (getsockname(sock, (struct sockaddr *)&server, &length) == -1)
	{
		perror("getsockname");
		exit(1);
	}
	
	printf("server ready waiting on port %d\n", ntohs(server.sin_port));
	
	/* We now are ready to wait for a basic TCP/IP connection up.
	 */

	listen(sock, 5);

	while (1)	/* Do this for each incoming TCP/IP connection */
	{
		/* Accept the new TCP/IP connection */
		if ((fd = accept(sock, NULL, NULL)) == -1)
		{
			perror("accept");
			exit(1);
		}

		/* Creates a new SSL connection.  This holds information
		 * pertinent to this
		 * connection.
		 */
		if ((s_ssl=SSL_new(s_ctx)) == NULL)
		{
			printf("SSL_new() failed\n");
			exit(1);
		}

		/* Tell SSL that this connection is to use the socket we 
		 * just created above. 
		 */
		SSL_set_fd(s_ssl, fd);

		/* Finally, start the SSL connection */
		if (SSL_accept(s_ssl)<1)
		{
			fprintf(stderr, "SSL_accept failed\n");
			ERR_print_errors_fp(stderr);

			SSL_free(s_ssl);
			close (fd);
			continue;
		}
		
		printf("starting connection using %s cipher\n", 
				SSL_get_cipher(s_ssl));
		
		do	/* Do this until the client disconnects: */
		{
			/* Receive data from the client and print it out */
			rval = SSL_read(s_ssl,buf,1024);
			if (rval < 0)
			{
				fprintf(stderr, "SSL_read: %s\n", ERR_reason_error_string(ERR_get_error()));
			}

			if (rval==0)
			{
				printf("ending connection\n");
			}				
			else
			{
				/* If everything is OK, print out data received */
				printf("-->%s\n", buf);

				/* Now send some data back to the client */
				SSL_write(s_ssl,"hello from server",18);
			}

		} while (rval>0);
		
		/* Close the SSL connection */
		SSL_free(s_ssl);

		/* Close the TCP/IP socket */
		close (fd);

	}
}
