 /*
  client.c

  To compile:
  
  cc -g -c -I/usr/local/SSLeay-0.6.4/include client.c
  cc -g client.o -L/usr/local/SSLeay-0.6.4/lib -lssl -lcrypto -lsocket -lnsl -o client

  This program implements a simple client which connects to the server by a TCP/IP connection, and then starts SSL on the connection.  It sends some data, then waits for some data (which it prints) and then disconnects.

  There are two arguments:
  	hostname to connect to
	port number (which the server will tell you when it starts).

  You will need to supply a certificate for a CA.  This is used in CAfile below.

  */

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include "buffer.h"
#include "crypto.h"
#include "../e_os.h"
#include "x509.h"
#include "ssl.h"
#include "err.h"

int MS_CALLBACK verify_callback(int ok, X509 *xs, X509 *xi, int depth, int error, char *arg);

#define CAfile "demoCA/cacert.pem"
#define CApath NULL

main(int argc, char **argv)
{
	int sock;		/* The TCP/IP socket */
	struct sockaddr_in server;
	struct hostent *hp;
	char buf[1024];
	
	SSL_CTX *c_ctx=NULL;	/* The Client's context */
	SSL *c_ssl=NULL;	/* The Client's SSL connection */
	int rval;
	
	if (argc<2)
	{
		printf("usage: client hostname port#\n");
		exit(1);
	}

	SSL_load_error_strings();

	/* Create a new context.  This holds information pertinent to the
	 * client's SSL side of the connection.
	 */
	c_ctx=SSL_CTX_new();

	if (c_ctx == NULL)
	{
		printf("SSL_CTX_new() failed\n");
	}

	/* Tell SSL where the Certificate Authority files are located */
	if ((!SSL_load_verify_locations(c_ctx,CAfile,CApath)) ||
	    (!SSL_set_default_verify_paths(c_ctx)))
	{
		fprintf(stderr,"SSL_load_verify_locations\n");
		ERR_print_errors_fp(stderr);
		exit(1);
	}

	/* Tell SSL to request the server's certificate when we connect. */
	SSL_CTX_set_verify(c_ctx,SSL_VERIFY_PEER, verify_callback);

	/* Now we can create a basic TCP/IP connection */
	if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0)
	{
		perror("socket");
		exit(1);
	}

	server.sin_family = AF_INET;

	if ((hp = gethostbyname(argv[1])) == NULL)
	{
		perror(argv[1]);
		exit(1);
	}

	memcpy((char *)&server.sin_addr, (char *)hp->h_addr, hp->h_length);
	server.sin_port = htons(atoi(argv[2]));
	if (connect(sock, (struct sockaddr *)&server, sizeof(server)) 
			== -1)
	{
		perror("connect");
		exit(1);
	}

	/* We now have a basic TCP/IP connection up.  Now we start SSL
	 * on this connection.
	 */

	/* Creates a new SSL connection.  This holds information pertinent
	 * to this connection.
	 */
	if ((c_ssl=SSL_new(c_ctx)) == NULL)
	{
		printf("SSL_new() failed\n");
		exit(1);
	}

	/* Tell SSL that this connection is to use the socket we just
	 * created above. 
	 */
	SSL_set_fd(c_ssl, sock);

	/* Finally, start the SSL connection */
	if (SSL_connect(c_ssl) < 1)
	{
		fprintf(stderr, "SSL_connect:");
		ERR_print_errors_fp(stderr);
		exit(1);
	}

	/* Lets find out who the peer *really* is.  We look though the
	 * server's certificate to see who he says he is.
	 */
	{
		X509 *peer_x509;
		char *s = NULL;

		peer_x509 = SSL_get_peer_certificate(c_ssl);

		if (peer_x509==0)
		{
			fprintf(stderr, "SSL_get_peer_cert:");
			ERR_print_errors_fp(stderr);
			exit(1);
		}

		s=(char *)X509_NAME_oneline(X509_get_subject_name(peer_x509));

		if (s==NULL)
		{
			fprintf(stderr, "X509_NAME_oneline:");
			ERR_print_errors_fp(stderr);
			exit(1);
		}

		printf("Server's subject name is '%s'\n', s);stderr
	}

	/* Send some data to the server */
	printf("sending data\n");
	SSL_write(c_ssl,"hello from client",18);

	memset(buf, 0, sizeof(buf));
	
	printf("waiting for data\n");

	/* Now we receive some data from the server and print it out */
	rval=SSL_read(c_ssl,buf,1024);
	printf("-->%s\n", buf);

	/* Close the SSL connection */
	SSL_free(c_ssl);

	/* Close the TCP/IP socket */
	close(sock);

	exit(0);
}

	
int MS_CALLBACK verify_callback(int ok, X509 *xs, X509 *xi, int depth, int error, char *arg)
{
	char *s;
	
	s=(char *)X509_NAME_oneline(X509_get_subject_name(xs));
	if (s != NULL)
	{
		if (ok)
			fprintf(stderr,"depth=%d %s\n",depth,s);
		else
		{
			fprintf(stderr,"depth=%d error=%d ok=%d %s\n",
					depth,error,ok,s);
			ERR_print_errors_fp(stderr);
		}
		Free(s);
	}
	
	return(ok);
}
