﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Net;
using System.Threading;
using Microsoft.WindowsAzure;
using Microsoft.WindowsAzure.Diagnostics;
using Microsoft.WindowsAzure.ServiceRuntime;
using Microsoft.WindowsAzure.StorageClient;
using System.IO;
using System.Text.RegularExpressions;

namespace ProcesorWideo
{
    public class WorkerRole : RoleEntryPoint
    {
        CloudQueue inputQueue;
        CloudQueue outputQueue;
        CloudStorageAccount csa;
        CloudBlobContainer processedAudio;
        static int showLatest = 0;
        public override void Run()
        {
            // This is a sample worker implementation. Replace with your logic.
            Trace.WriteLine("ProcesorWideo rozpocząć działanie", "Information");

            while (true)
            {
                //sprawdzenie czy kolejka zawiera pliki do przetworzenia
                CloudQueueMessage receivedMessage = inputQueue.GetMessage();

                //jeżeli komunikat z kolejki nie pusty
                if (receivedMessage != null)
                {
                    //wyodrębnienie poszczególne elementy komunikatu
                    string[] message = receivedMessage.AsString.Split('|');
                    string audioGuid = message[0];
                    string link = message[1];
                    //wyodrębnienie nazwy pliku z dostarczonego linka, potrzebne przy późniejszym dodawaniu informacji do Azure Tables
                    Regex fileNameRegex = new Regex("(?<=(&title=)).*", RegexOptions.RightToLeft);
                    string fileName = fileNameRegex.Match(link).Value;

                    //tworzenie identyfikatora GUID dla pobieranego do pamięci lokalnej pliku wideo
                    Guid videoGuid = Guid.NewGuid();
                    //przygotowanie ścieżek dostępu do plików wejściowego i wyjściowego
                    string localVideoFilePath = Path.Combine(RoleEnvironment.GetLocalResource("LocalStorage1").RootPath, videoGuid.ToString() + ".mp4");
                    string localAudioFilePath = Path.Combine(RoleEnvironment.GetLocalResource("LocalStorage1").RootPath, audioGuid + ".mp3");
                    try
                    {
                        //pobieranie pliku wideo do pamięci lokalnej
                        WebClient webClient = new WebClient();
                        webClient.DownloadFile(link, localVideoFilePath);
                    }
                    catch (Exception)
                    {
                        Trace.WriteLine("Błąd przy ściąganiu pliku wideo na dysk lokalny: ");
                    }

                    //przygotowanie ścieżki do pliku wykonywalnego ffmpeg
                    string path = Path.Combine(Environment.GetEnvironmentVariable("RoleRoot"), "approot", "ffmpeg", "ffmpeg.exe");

                    //przygotowanie obiektu procesu
                    ProcessStartInfo psi = new ProcessStartInfo();
                    //Przełączniki, argumenty i ścieżki dla kodeka
                    psi.FileName = path;
                    psi.Arguments = string.Format(@"-i ""{0}"" -f mp3 -ab 192000 -vn ""{1}""", localVideoFilePath, localAudioFilePath);
                    psi.CreateNoWindow = true;
                    psi.ErrorDialog = false;
                    psi.UseShellExecute = false;

                    try
                    {
                        using (Process exeProcess = Process.Start(psi))
                        {
                            //uruchomienie procesu konwersji
                            exeProcess.WaitForExit();
                        }
                    }
                    catch (Exception ex)
                    {
                        Trace.WriteLine("Błąd przy wyłuskiwaniu strumienia audio: " + ex.Message);
                    }
                    try
                    {
                        //przygotowanie ścieżki wyjściowego pliku MP3
                        string blobFileName = Path.GetFileName(localAudioFilePath);
                        //budowanie obiektu bloba dla wyjściowego pliku
                        CloudBlob cb = processedAudio.GetBlobReference(blobFileName);
                        //wysłanie bloba do Azure Storage (lub Development Storage)
                        cb.UploadFile(localAudioFilePath);
                        //koniec przetwarzania
                        //usuwanie komunikatu z kolejki wejściowej
                        inputQueue.DeleteMessage(receivedMessage);

                        //utworzenie kontekstu dostępu do tabeli w Azure Tables
                        var context = new PrzetworzoneMP3DataServiceContext(csa.TableEndpoint.ToString(), csa.Credentials);
                        //przygotowanie encji do wstawienia
                        var entity = new PrzetworzoneMP3()
                        {
                            Nazwa = fileName,
                            Link = cb.Uri.AbsoluteUri.ToString()
                        };

                        //pobranie wszystkich rekordów
                        IEnumerable<PrzetworzoneMP3> query = context.PrzetworzoneMP3Table;
                        //jeśli więcej niż ustalona wartość usuwanie ostatniego
                        if (query.Count<PrzetworzoneMP3>() > showLatest - 1)
                        {
                            PrzetworzoneMP3 toDelete = null;
                            string blobToDelete = null;
                            //pobranie informacji o najstarszym przetworzonym pliku
                            toDelete = query.OrderBy(x => x.Timestamp).First();
                            //zapamiętanie odsyłacza do bloba, którego należy usunąć
                            blobToDelete = toDelete.Link;
                            //usunięcie obiektu z Tables
                            context.DeleteObject(toDelete);
                            //zapamiętanie zmian
                            context.SaveChanges();
                            //przygotowanie informacji o blobie
                            CloudBlob cblob = processedAudio.GetBlobReference(blobToDelete);
                            //usunięcie bloba z Azure Blobs
                            cblob.Delete();
                        }
                        //dodanie obiektu do kontekstu i zapis zmian
                        context.AddObject("last", entity);
                        context.SaveChanges();


                        //przygotowanie komunikatu do kolejki wyjściowej
                        CloudQueueMessage mes = new CloudQueueMessage(audioGuid + "|" + cb.Uri.AbsoluteUri);
                        //wysłanie komunikatu do kolejki wyjściowej
                        outputQueue.AddMessage(mes);
                    }
                    catch (Exception ex)
                    {
                        Trace.WriteLine("Błąd przy wysyłaniu przetworzonego audio do Azure Blobs: " + ex.Message);

                    }
                    finally
                    {
                        //końcowe usuwanie plików roboczych
                        File.Delete(localVideoFilePath);
                        File.Delete(localAudioFilePath);
                    }
                }
                //uśpienie procesu na sekundę
                Thread.Sleep(1000);
                Trace.WriteLine("Pracuję", "Information");
            }
        }

        public override bool OnStart()
        {
            //pobranie z pliku konfiguracyjnego wartości parametru showLatest
            showLatest = int.Parse(RoleEnvironment.GetConfigurationSettingValue("showLatest"));
            //Utworzenie obiektu podstawowego komunikacji z Azure Storage
            //CloudStorageAccount wskazuje na lokalny emulator Azure Storage
            csa = CloudStorageAccount.Parse(RoleEnvironment.GetConfigurationSettingValue("StorageCredentials"));
            //utworzenie klienta Azure Tables na podstawie CloudStorageAccount
            CloudTableClient ctc = csa.CreateCloudTableClient();
            //Metoda CreateIfNotExist gwarantuje utworzenie tabeli
            ctc.CreateTableIfNotExist("last");
            //utworzenie klienta Azure Blobs na podstawie CloudStorageAccount
            CloudBlobClient cbc = csa.CreateCloudBlobClient();
            //utworzenie klienta Azure Queues na podstawie CloudStorageAccount
            CloudQueueClient cqc = csa.CreateCloudQueueClient();
            //pobranie referencji do kolejek
            inputQueue = cqc.GetQueueReference("inputqueue");
            outputQueue = cqc.GetQueueReference("outputqueue");
            processedAudio = cbc.GetContainerReference("processed");
            //utworzenie kolejek 
            outputQueue.CreateIfNotExist();
            inputQueue.CreateIfNotExist();
            //przygotowanie praw dostępu do kontenera blobów
            BlobContainerPermissions bcp = new BlobContainerPermissions();
            bcp.PublicAccess = BlobContainerPublicAccessType.Blob;
            //utworzenie kontenera
            processedAudio.CreateIfNotExist();
            //nadanie praw dostępu kontenerowi
            processedAudio.SetPermissions(bcp);

            ServicePointManager.DefaultConnectionLimit = 12;

            return base.OnStart();
        }
    }
}
