'use strict';

exports.__esModule = true;
exports.default = tokenize;
var SINGLE_QUOTE = 39;
var DOUBLE_QUOTE = 34;
var BACKSLASH = 92;
var SLASH = 47;
var NEWLINE = 10;
var SPACE = 32;
var FEED = 12;
var TAB = 9;
var CR = 13;
var OPEN_PARENTHESES = 40;
var CLOSE_PARENTHESES = 41;
var OPEN_CURLY = 123;
var CLOSE_CURLY = 125;
var SEMICOLON = 59;
var ASTERICK = 42;
var COLON = 58;
var AT = 64;

var RE_AT_END = /[ \n\t\r\f\{\(\)'"\\;/]/g;
var RE_WORD_END = /[ \n\t\r\f\(\)\{\}:;@!'"\\]|\/(?=\*)/g;
var RE_BAD_BRACKET = /.[\\\/\("'\n]/;

function tokenize(input) {
    var tokens = [];
    var css = input.css.valueOf();

    var code = void 0,
        next = void 0,
        quote = void 0,
        lines = void 0,
        last = void 0,
        content = void 0,
        escape = void 0,
        nextLine = void 0,
        nextOffset = void 0,
        escaped = void 0,
        escapePos = void 0,
        prev = void 0,
        n = void 0;

    var length = css.length;
    var offset = -1;
    var line = 1;
    var pos = 0;

    function unclosed(what) {
        throw input.error('Unclosed ' + what, line, pos - offset);
    }

    while (pos < length) {
        code = css.charCodeAt(pos);

        if (code === NEWLINE || code === FEED || code === CR && css.charCodeAt(pos + 1) !== NEWLINE) {
            offset = pos;
            line += 1;
        }

        switch (code) {
            case NEWLINE:
            case SPACE:
            case TAB:
            case CR:
            case FEED:
                next = pos;
                do {
                    next += 1;
                    code = css.charCodeAt(next);
                    if (code === NEWLINE) {
                        offset = next;
                        line += 1;
                    }
                } while (code === SPACE || code === NEWLINE || code === TAB || code === CR || code === FEED);

                tokens.push(['space', css.slice(pos, next)]);
                pos = next - 1;
                break;

            case OPEN_CURLY:
                tokens.push(['{', '{', line, pos - offset]);
                break;

            case CLOSE_CURLY:
                tokens.push(['}', '}', line, pos - offset]);
                break;

            case COLON:
                tokens.push([':', ':', line, pos - offset]);
                break;

            case SEMICOLON:
                tokens.push([';', ';', line, pos - offset]);
                break;

            case OPEN_PARENTHESES:
                prev = tokens.length ? tokens[tokens.length - 1][1] : '';
                n = css.charCodeAt(pos + 1);
                if (prev === 'url' && n !== SINGLE_QUOTE && n !== DOUBLE_QUOTE && n !== SPACE && n !== NEWLINE && n !== TAB && n !== FEED && n !== CR) {
                    next = pos;
                    do {
                        escaped = false;
                        next = css.indexOf(')', next + 1);
                        if (next === -1) unclosed('bracket');
                        escapePos = next;
                        while (css.charCodeAt(escapePos - 1) === BACKSLASH) {
                            escapePos -= 1;
                            escaped = !escaped;
                        }
                    } while (escaped);

                    tokens.push(['brackets', css.slice(pos, next + 1), line, pos - offset, line, next - offset]);
                    pos = next;
                } else {
                    next = css.indexOf(')', pos + 1);
                    content = css.slice(pos, next + 1);

                    if (next === -1 || RE_BAD_BRACKET.test(content)) {
                        tokens.push(['(', '(', line, pos - offset]);
                    } else {
                        tokens.push(['brackets', content, line, pos - offset, line, next - offset]);
                        pos = next;
                    }
                }

                break;

            case CLOSE_PARENTHESES:
                tokens.push([')', ')', line, pos - offset]);
                break;

            case SINGLE_QUOTE:
            case DOUBLE_QUOTE:
                quote = code === SINGLE_QUOTE ? '\'' : '"';
                next = pos;
                do {
                    escaped = false;
                    next = css.indexOf(quote, next + 1);
                    if (next === -1) unclosed('quote');
                    escapePos = next;
                    while (css.charCodeAt(escapePos - 1) === BACKSLASH) {
                        escapePos -= 1;
                        escaped = !escaped;
                    }
                } while (escaped);

                content = css.slice(pos, next + 1);
                lines = content.split('\n');
                last = lines.length - 1;

                if (last > 0) {
                    nextLine = line + last;
                    nextOffset = next - lines[last].length;
                } else {
                    nextLine = line;
                    nextOffset = offset;
                }

                tokens.push(['string', css.slice(pos, next + 1), line, pos - offset, nextLine, next - nextOffset]);

                offset = nextOffset;
                line = nextLine;
                pos = next;
                break;

            case AT:
                RE_AT_END.lastIndex = pos + 1;
                RE_AT_END.test(css);
                if (RE_AT_END.lastIndex === 0) {
                    next = css.length - 1;
                } else {
                    next = RE_AT_END.lastIndex - 2;
                }
                tokens.push(['at-word', css.slice(pos, next + 1), line, pos - offset, line, next - offset]);
                pos = next;
                break;

            case BACKSLASH:
                next = pos;
                escape = true;
                while (css.charCodeAt(next + 1) === BACKSLASH) {
                    next += 1;
                    escape = !escape;
                }
                code = css.charCodeAt(next + 1);
                if (escape && code !== SLASH && code !== SPACE && code !== NEWLINE && code !== TAB && code !== CR && code !== FEED) {
                    next += 1;
                }
                tokens.push(['word', css.slice(pos, next + 1), line, pos - offset, line, next - offset]);
                pos = next;
                break;

            default:
                if (code === SLASH && css.charCodeAt(pos + 1) === ASTERICK) {
                    next = css.indexOf('*/', pos + 2) + 1;
                    if (next === 0) unclosed('comment');

                    content = css.slice(pos, next + 1);
                    lines = content.split('\n');
                    last = lines.length - 1;

                    if (last > 0) {
                        nextLine = line + last;
                        nextOffset = next - lines[last].length;
                    } else {
                        nextLine = line;
                        nextOffset = offset;
                    }

                    tokens.push(['comment', content, line, pos - offset, nextLine, next - nextOffset]);

                    offset = nextOffset;
                    line = nextLine;
                    pos = next;
                } else {
                    RE_WORD_END.lastIndex = pos + 1;
                    RE_WORD_END.test(css);
                    if (RE_WORD_END.lastIndex === 0) {
                        next = css.length - 1;
                    } else {
                        next = RE_WORD_END.lastIndex - 2;
                    }

                    tokens.push(['word', css.slice(pos, next + 1), line, pos - offset, line, next - offset]);
                    pos = next;
                }

                break;
        }

        pos++;
    }

    return tokens;
}
module.exports = exports['default'];
//# sourceMappingURL=data:application/json;base64,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
