﻿using System.IO;
using System.Windows.Controls;
using System.Windows.Media;
using Microsoft.VisualStudio.Text.Editor;

namespace CodeMetricAdornment
{
    /// <summary>
    /// Klasa dekoracji wyświetlająca prostokątne pole 
    /// w prawym górnym rogu okna.
    /// </summary>
    class CodeMetricAdornment
    {
        private IWpfTextView _view;
        private IAdornmentLayer _adornmentLayer;
        private CodeMetricDisplayControl _displayControl = 
            new CodeMetricDisplayControl();

        /// <summary>
        /// Tworzy prostokątny rysunek i dołącza metodę obsługi zdarzenia LayoutChanged,
        /// która umieszcza grafikę w prawym górnym rogu obiektu TextView przy użyciu warstwy dekoracji.
        /// </summary>
        /// <param name="view">Obiekt <see cref="IWpfTextView"/>, na którym narysowany zostanie obrazek.</param>
        public CodeMetricAdornment(IWpfTextView view)
        {
            _view = view;

            _displayControl.LinesOfCode = CountLOC(view);
            _displayControl.CommentLines = CountCommentLines(view);
            _displayControl.WhitespaceLines = CountWhitespaceLines(view);


            // Pobieranie referencji do warstwy, w której należy umieścić daną dekorację.
            _adornmentLayer = view.GetAdornmentLayer("CodeMetricAdornment");

            _view.LayoutChanged += this.OnLayoutChanged;
            _view.ViewportHeightChanged += delegate { this.onSizeChange(); };
            _view.ViewportWidthChanged += delegate { this.onSizeChange(); };
        }

        public void onSizeChange()
        {
            // Usuwanie wcześniejszych dekoracji z ich warstwy.
            _adornmentLayer.RemoveAllAdornments();

            int buffer = 50;

            // Umieszczanie rysunku w prawym górnym rogu okna.
            Canvas.SetLeft(_displayControl, 
                _view.ViewportRight - (_displayControl.ActualWidth
                + buffer));
            Canvas.SetTop(_displayControl, 
                _view.ViewportTop + (_displayControl.ActualHeight
                + buffer));

            // Dodawanie rysunku do warstwy dekoracji i określanie jego
            // lokalizacji względem okna.
            _adornmentLayer.AddAdornment(
                AdornmentPositioningBehavior.ViewportRelative, 
                null, null, _displayControl, null);
        }

        private void OnLayoutChanged(object sender, TextViewLayoutChangedEventArgs e)
        {
            _displayControl.LinesOfCode = CountLOC(_view);
            _displayControl.CommentLines = CountCommentLines(_view);
            _displayControl.WhitespaceLines = CountWhitespaceLines(_view);
        }

        private int CountLOC(IWpfTextView view)
        {
           string code = view.TextSnapshot.GetText();

           int count = 1;
           int start = 0;
           while ((start = code.IndexOf('\n', start)) != -1)
           {
               count++;
               start++;
           }
           
            return count;

        }

        private int CountWhitespaceLines(IWpfTextView view)
        {
            string code = view.TextSnapshot.GetText();
            int count = 0;

            using (StringReader reader = new StringReader(code)) 
            { 
                string line; 
                
                while ((line = reader.ReadLine()) != null) 
                {
                    if (line.Trim() == "")
                        count++;
                }

                return count;

            } 
        }

        private int CountCommentLines(IWpfTextView view)
        {
            string code = view.TextSnapshot.GetText();
            int count = 0;

            using (StringReader reader = new StringReader(code))
            {
                string line;

                while ((line = reader.ReadLine()) != null)
                {
                    if (line.TrimStart().StartsWith("//"))
                        count++;
                }

                return count;

            }
        }
    }
}
