﻿// **** Autor: Jacek Ross - games@ejr.com.pl
// **** Plik jest czescia projektu SymulatorZielarza stworzonego na potrzeby ksiazki "Unity i C#. Praktyka programowania gier" wyd. Helion 2020 oraz czescia kodu gry SymulatorZielarza wydanej w serwisie Steam, wiecej info: http://ejr.com.pl
// **** Dozwolone użycie pod licencją CC0 1.0 (public domain). 
// **** Jeżeli modyfikujesz niniejszy plik - umiesc w nim informacje o tym, ze zawartosc odbiega od oryginalu albo usun informacje o autorze

// **** Author: Jacek Ross - games@ejr.com.pl
// **** This file is a part of SymulatorZielarza project created for book "Unity i C#. Praktyka programowania gier" published by Helion 2020 and a part of source of game SymulatorZielarza published on Steam, more info: http://ejr.com.pl
// **** Usage permission under CC0 1.0 license (public domain). 
// **** If you modified this file - put here an information about that the content of file is modified and diffrent than the original, or delete info about original author


using System.Collections.Generic;
using System.Text;
using UnityEngine;

namespace ZielarzSilnik
{         
    public class Pacjent: Postac
    {

        public const float MINIMUM_ZDROWIA_WITALNOSCI_NASTART = 10;
        public const float MINIMUM_WAZNEGO_ZDARZENIA = 10;
        
        public bool JestZywy {  get { return PobierzAtrybut("Zdrowie") > 0; } }
        public bool JestZdrowy { get { return _aktualneZdarzenia.Count == 0; } }
        public string PrzyczynaZgonu { get; private set; }
        public int LiczbaAktualnychZdarzen {  get { return _aktualneZdarzenia.Count; } }
        public int LiczbaLeczacychPrzedmiotow {  get { return _przedmiotyLeczace.Count; } }
        public int LicznikOczekiwania { get; private set; }         //okresla ile jeszcze godzin pacjent bedzie czekac na obslizenie, liczba -1 oznacza ze nieskonczonosc
        public string ImieNazwisko { get; set; }
        public List<string> WazneZdarzenia { get; private set; }          //istotne zdarzenie ktore zaszlo w trakcie obslugi godzinowej

        private List<PrzedmiotLeczniczy> _przedmiotyLeczace;        // lista przedmiotow akutalnie leczacych pacjenta
        private List<ZdarzenieMedyczne> _aktualneZdarzenia;         // lista aktualnych zdarzen medycznych
        private List<CzescCiala> _wylaczoneLokalizacje;       //lista lokalizacji dla ktorych nie mozna juz utworzyc zdarzen medycznych bo... dana czesc ciala nie funkcjonuje calkowicie (np. amputowana reka nie moze zostac odmrozona)      
        private bool _pacjentLezacy;

        public Pacjent(int poziomTrudnosci) : base()
        {
            float zdrowie = 100 - Random.Range(0, poziomTrudnosci * 5f);
            if (zdrowie < MINIMUM_ZDROWIA_WITALNOSCI_NASTART)
                zdrowie = MINIMUM_ZDROWIA_WITALNOSCI_NASTART;
            float witalnosc = Random.Range(50, 101) - Random.Range(0, poziomTrudnosci * 5f);
            if (witalnosc < MINIMUM_ZDROWIA_WITALNOSCI_NASTART)
                witalnosc = MINIMUM_ZDROWIA_WITALNOSCI_NASTART;

            UtworzAtrybut("Zdrowie", zdrowie, -100, 100);          //ogolny poziom zycia, po spadku do 0 - pacjent umiera z powodu niewydolnosci ogolnej organizmu
            UtworzAtrybut("Sila", 100, 0, 100);              //sila miesni
            UtworzAtrybut("Witalnosc", witalnosc, 1, 100);    //poziom odpornosci oraz zdolnosci regenracji, sila zdarzenia zmniejsza sie co godzine o (Witalnosc/50) % swojej wartosci oraz -0.01 * Witalnosc/100 na godz
            WazneZdarzenia = new List<string>();
            _aktualneZdarzenia = new List<ZdarzenieMedyczne>();
            _wylaczoneLokalizacje = new List<CzescCiala>();
            _przedmiotyLeczace = new List<PrzedmiotLeczniczy>();
            PrzyczynaZgonu = "";
            LicznikOczekiwania = -1;
            _pacjentLezacy = false;
            ImieNazwisko = "?";
        }

        //dodaje nowe zdarzenie pierwotne (nowe choroby)
        public void DodajNoweZdarzenie(ZdarzenieMedyczne zdarzenie)
        {
            _aktualneZdarzenia.Add(zdarzenie);
        }

        //obsluga zdarzen medycznych godzinowo
        public void GodzinowaObslugaZdarzen()
        {
            if (LicznikOczekiwania > 0)
                LicznikOczekiwania--;
            WazneZdarzenia.Clear();
            List<ZdarzenieMedyczne> wszystkieNoweZdarzenia = new List<ZdarzenieMedyczne>();  //skumulowana lista nowych zdarzen medycznych
            List<ZdarzenieMedyczne> zdarzeniaDoUsuniecia = new List<ZdarzenieMedyczne>();     //zdarzenia do usuniecia
            foreach(ZdarzenieMedyczne zdarzenie in _aktualneZdarzenia)
            {
                List<ZdarzenieMedyczne> noweZdarzenia = MenedzerZielarstwa.Instancja.PrzetworzStanZdarzeniaMedycznego(zdarzenie);
                wszystkieNoweZdarzenia.AddRange(noweZdarzenia);
                //obsluzmy jeszcze witalnosc pacjenta
                zdarzenie.Sila = zdarzenie.Sila * (1 - PobierzAtrybut("Witalnosc") / 5000f) - 0.01f * (PobierzAtrybut("Witalnosc") / 50f);
                //oraz leczenie przedmiotami
                float leczenie = PodajGodzinoweLeczeniePrzedmiotem(zdarzenie);
                zdarzenie.Sila -= 1f * leczenie;
                zdarzenie.Zaostrzenie -= 0.5f * leczenie;
                //obsluz wplyw zdarzenia na parametry ogolne
                ObsluzWplywGodzinowyZdarzeniaNaParametry(zdarzenie);
                //smiertelnosc gwaltowna zdarzenia
                if ( zdarzenie.Smiertelnosc > 0 && Random.Range(0f, 1f) < (1.5f * zdarzenie.Smiertelnosc * MechanikaCzesciCiala.MnoznikCzesciCiala(zdarzenie.Lokalizacja)) && Random.Range(0f, 100f) > PobierzAtrybut("Witalnosc") * 0.75f)
                {
                    PrzyczynaZgonu = string.Format("{0}", zdarzenie.DajLadnaNazwe());
                    UstawAtrybut("Zdrowie", -100);
                }
                //zdarzenie w wyniku przetworzenia moglo oslabnac do zera - do usuniecia
                if (zdarzenie.Sila <= zdarzenie.MinimalnaSila)
                    zdarzeniaDoUsuniecia.Add(zdarzenie);
                //zdarzenie jednorazowe bedzie usuniete po przetworzeniu
                if (zdarzenie.Jednorazowe)
                    zdarzeniaDoUsuniecia.Add(zdarzenie);
                zdarzenie.Normalizuj();
            }
            //efekty uboczne przedmiotow
            List<ZdarzenieMedyczne> zdarzeniaUboczne = PodajGodzinoweZdarzeniaUbocznePrzedmiotow();
            _aktualneZdarzenia.AddRange(zdarzeniaUboczne);
            _aktualneZdarzenia.AddRange(wszystkieNoweZdarzenia);
            //wplyw na parametry ogolne od przedmiotow
            ObsluzGodzinowyWplywOgolnyPrzedmiotow();
            //nowe zdarzenia powinny być osłabione o działanie aktualnych leków (daje to działanie zapobiegawcze)
            foreach (ZdarzenieMedyczne zdarzenie in wszystkieNoweZdarzenia)
            {
                zdarzenie.Sila -= PodajGodzinoweLeczeniePrzedmiotem(zdarzenie);
                if (zdarzenie.Sila <= zdarzenie.MinimalnaSila)
                    _aktualneZdarzenia.Remove(zdarzenie);
            }
            //musimy jeszcze przejrzec liste aby polaczyc zdarzenia, ktore moga byc identyczne i dotyczyc tego samego miejsca (stworzone z dwoch roznych chorob) np. dwa zapalenia gardla redukuja sie do jednego mocniejszego
            for(int i = 0; i < _aktualneZdarzenia.Count; i++)
                for (int j = i + 1; j < _aktualneZdarzenia.Count; j++)
                {
                    if(_aktualneZdarzenia[i].Nazwa == _aktualneZdarzenia[j].Nazwa && (_aktualneZdarzenia[i].Lokalizacja & _aktualneZdarzenia[j].Lokalizacja) != 0)
                    {
                        //zdarzenie bedzie suma sil zdarzen i najwiekszym zaostrzeniem, a zdarzenie2 zostanie usuniete
                        _aktualneZdarzenia[i].Sila = _aktualneZdarzenia[i].Sila + _aktualneZdarzenia[j].Sila;
                        _aktualneZdarzenia[i].Zaostrzenie = _aktualneZdarzenia[i].Zaostrzenie > _aktualneZdarzenia[j].Zaostrzenie ? _aktualneZdarzenia[i].Zaostrzenie : _aktualneZdarzenia[j].Zaostrzenie;
                        _aktualneZdarzenia[i].Normalizuj();
                        zdarzeniaDoUsuniecia.Add(_aktualneZdarzenia[j]);
                    }
                }
            //obsluga amputacji 
            foreach (ZdarzenieMedyczne zdarzenie in _aktualneZdarzenia)
                if (zdarzenie.Nazwa == "Amputacja" && !_wylaczoneLokalizacje.Contains(zdarzenie.Lokalizacja))
                {
                    _wylaczoneLokalizacje.Add(zdarzenie.Lokalizacja);
                }
            foreach (ZdarzenieMedyczne zdarzenie in _aktualneZdarzenia)
                if (_wylaczoneLokalizacje.Contains(zdarzenie.Lokalizacja))
                    zdarzeniaDoUsuniecia.Add(zdarzenie);
            //usuniecie niepotrzebnych zdarzen
            foreach (ZdarzenieMedyczne zdarzenie in zdarzeniaDoUsuniecia)
                _aktualneZdarzenia.Remove(zdarzenie);
            ObsluzZuzycieGodzinowePrzedmiotow();
            SortujZdarzenia();
            if (_pacjentLezacy && !CzyJestMocnoChory() && LicznikOczekiwania == -1)
                LicznikOczekiwania = KonfiguratorGry.MaksimumOczekiwaniaPacjenta;
        }
        //losuje choroby tak dlugo az wylosuje conajmniej rowna poziomTrudnosci / 2, ale nie mniej niz 100 losowan i nie wiecej niz 1000 losowan, a takze nie wiecej chorob niz MAKSIMUM_CHOROB
        public void WylosujChoroby(bool krytyczne, int poziomTrudnosci)
        {
            int minimalnaLiczbaZdarzen = poziomTrudnosci < 2 ? 1 : poziomTrudnosci / 2;
            int liczbaLosowan = 0;
            do
            {
                ZdarzenieMedyczne zdarzenie = MenedzerZielarstwa.Instancja.UtworzPierwotneZdarzenieMedyczne();
                if (zdarzenie != null && (! krytyczne || zdarzenie.Sila >= 0.5f) && (krytyczne || zdarzenie.Sila <= 0.5f))
                    DodajNoweZdarzenie(zdarzenie);
                liczbaLosowan++;
            } while (_aktualneZdarzenia.Count < KonfiguratorGry.MaksimumChorobPacjenta && _aktualneZdarzenia.Count < minimalnaLiczbaZdarzen && (liczbaLosowan < 100 || (liczbaLosowan < 1000 && _aktualneZdarzenia.Count == 0)));
            SortujZdarzenia();
            if (krytyczne)
            {
                LicznikOczekiwania = -1;
                _pacjentLezacy = true;
            }
            else
            {
                LicznikOczekiwania = KonfiguratorGry.MaksimumOczekiwaniaPacjenta;
                _pacjentLezacy = false;
            }
        }

        public void DodajPrzedmiotLeczacy(PrzedmiotLeczniczy nowyPrzedmiot)
        {
            nowyPrzedmiot.ResetujCzasDzialania();
            _przedmiotyLeczace.Add(nowyPrzedmiot);
        }

        public string ZrzutZdarzen(bool wersjaDebug)
        {
            StringBuilder log = new StringBuilder();
            log.Append(string.Format("{0}, {1}: {2}, {3}: {4} \n\r",
                 ImieNazwisko, TlumaczCiagow.PodajCiag("ZdrowieSkrot"), ZrobTekst(PobierzAtrybut("Zdrowie")), TlumaczCiagow.PodajCiag("WitalnoscSkrot"), ZrobTekst(PobierzAtrybut("Witalnosc"))));
            foreach (PrzedmiotLeczniczy przedmiotL in _przedmiotyLeczace)
                log.Append(string.Format(" % {0} {1} \n\r", TlumaczCiagow.PodajCiag("PrzedmiotLeczacy"), TlumaczCiagow.PodajCiag("Przedmiot" + przedmiotL.Nazwa)));
            foreach (CzescCiala wykluczonaCzescCiala in _wylaczoneLokalizacje)
                log.Append(string.Format(" # {0}: {1} \n\r", TlumaczCiagow.PodajCiag("Amputowano"), wykluczonaCzescCiala));
            if (wersjaDebug)
                foreach (ZdarzenieMedyczne zdarzenie in _aktualneZdarzenia)
                    log.Append(" * " + zdarzenie.DajLadnaNazwe() + " " + ZrobTekst(zdarzenie.Sila) + " (" + ZrobTekst(zdarzenie.Zaostrzenie) + ")\n\r");
            else
                foreach (ZdarzenieMedyczne zdarzenie in _aktualneZdarzenia)
                    log.Append(" * " + zdarzenie.DajLadnaNazwe() + ", " + TlumaczCiagow.PodajCiag("Zagrozenie") + zdarzenie.StanSily() + "\n\r");
            return log.ToString();
        }
        public ZdarzenieMedyczne PodajZdarzenie(int i)
        {
            if (i >= 0 && i < LiczbaAktualnychZdarzen)
                return _aktualneZdarzenia[i];
            return null;
        }

        //po odejsciu od domu zielarza
        public void KoniecLeczenia()
        {
            if (_pacjentLezacy)
                MenedzerGry.InstancjaMenedzeraGry.ObiektGracza.UstawAtrybut("Zloto", MenedzerGry.InstancjaMenedzeraGry.ObiektGracza.PobierzAtrybut("Zloto") + KonfiguratorGry.HonorariumZaLeczenieSzpitalne);
            while (JestZywy && !JestZdrowy)
               GodzinowaObslugaZdarzen();
                  
        }
        //po wyzdrowieniu
        public void PacjentWyzdrowial()
        {
            MenedzerGry.InstancjaMenedzeraGry.ObiektGracza.UstawAtrybut("Zloto", MenedzerGry.InstancjaMenedzeraGry.ObiektGracza.PobierzAtrybut("Zloto") + KonfiguratorGry.HonorariumZaWyleczenie +
               (int)(KonfiguratorGry.HonorariumZaWyleczenie * MenedzerGry.InstancjaMenedzeraGry.ObiektGracza.PobierzAtrybut("Reputacja")));
            MenedzerGry.InstancjaMenedzeraGry.ObiektGracza.UstawAtrybut("Reputacja", MenedzerGry.InstancjaMenedzeraGry.ObiektGracza.PobierzAtrybut("Reputacja") * KonfiguratorGry.ReputacjaZaWyleczenie);
            MenedzerGry.InstancjaMenedzeraGry.ZapiszZmienna("Wyzdrowieli", MenedzerGry.InstancjaMenedzeraGry.OdczytajZmiennaInt("Wyzdrowieli") + 1);
        }
        //po zgonie
        public void PacjentZmarl()
        {
            MenedzerGry.InstancjaMenedzeraGry.ObiektGracza.UstawAtrybut("Reputacja", MenedzerGry.InstancjaMenedzeraGry.ObiektGracza.PobierzAtrybut("Reputacja") * KonfiguratorGry.ReputacjaZaSmierc);
            MenedzerGry.InstancjaMenedzeraGry.ZapiszZmienna("Zmarli", MenedzerGry.InstancjaMenedzeraGry.OdczytajZmiennaInt("Zmarli") + 1);
        }

        //zwraca true jeśli conajmniej jedno zdarzenie medyczne ma sile > 0.5
        private bool CzyJestMocnoChory()
        {
            for (int i = 0; i < LiczbaAktualnychZdarzen; i++)
                if (_aktualneZdarzenia[i].Sila > 0.5f)
                    return true;
            return false;
        }

        private string ZrobTekst(float liczba)
        {
            return liczba.ToString("F2");
        }
        private float PodajGodzinoweLeczeniePrzedmiotem(ZdarzenieMedyczne zdarzenie)
        {
            foreach (PrzedmiotLeczniczy przedmiot in _przedmiotyLeczace)
            {
                float leczenie = przedmiot.PodajGodzinoweLeczenie(zdarzenie);
                if (leczenie != 0)
                    return leczenie;
            }
            return 0;
        }
        private List<ZdarzenieMedyczne> PodajGodzinoweZdarzeniaUbocznePrzedmiotow()
        {
            List<ZdarzenieMedyczne> uboczne = new List<ZdarzenieMedyczne>();
            foreach (PrzedmiotLeczniczy przedmiot in _przedmiotyLeczace)
                uboczne.AddRange(przedmiot.WylosujZdarzenieNiepozadane());
            return uboczne;
        }
        private void ObsluzGodzinowyWplywOgolnyPrzedmiotow()
        {
            foreach (PrzedmiotLeczniczy przedmiot in _przedmiotyLeczace)
            {
                List<WplywNaParametr> wplywPrzedmiotu = przedmiot.PodajOgolnyWplyw();
                foreach (WplywNaParametr wplyw in wplywPrzedmiotu)
                    UstawAtrybut(wplyw.Parametr, PobierzAtrybut(wplyw.Parametr) + 1f * wplyw.Wartosc );               
            }
        }
        private void ObsluzZuzycieGodzinowePrzedmiotow()
        {
            List<PrzedmiotLeczniczy> doUsuniecia = new List<PrzedmiotLeczniczy>();
            foreach(PrzedmiotLeczniczy przedmiot in _przedmiotyLeczace)
            {
                przedmiot.ZuzyjDawke();
                if (przedmiot.PozostalyCzasDzialania <= 0)
                    doUsuniecia.Add(przedmiot);
            }
            foreach (PrzedmiotLeczniczy przedmiot in doUsuniecia)
                _przedmiotyLeczace.Remove(przedmiot);
        }
        private void ObsluzWplywGodzinowyZdarzeniaNaParametry(ZdarzenieMedyczne zdarzenie)
        {
            foreach (WplywNaParametr wplyw in zdarzenie.SzkodliwoscOgolna)
            {
                float wielkoscWplywu =  wplyw.Wartosc * zdarzenie.Sila * MechanikaCzesciCiala.MnoznikCzesciCiala(zdarzenie.Lokalizacja);
                if (wielkoscWplywu <= -MINIMUM_WAZNEGO_ZDARZENIA && wplyw.Parametr == "Zdrowie")
                    WazneZdarzenia.Add(TlumaczCiagow.PodajCiag("PowaznyWplyw") + zdarzenie.DajLadnaNazwe());
                UstawAtrybut(wplyw.Parametr, PobierzAtrybut(wplyw.Parametr) + wielkoscWplywu);
            }
        }
        private void SortujZdarzenia()
        {
            for(int i = 0; i < LiczbaAktualnychZdarzen; i++)
                for (int j =  i + 1; j < LiczbaAktualnychZdarzen; j++)
                    if(_aktualneZdarzenia[i].Sila < _aktualneZdarzenia[j].Sila)
                    {
                        ZdarzenieMedyczne tymczasowe = _aktualneZdarzenia[i];
                        _aktualneZdarzenia[i] = _aktualneZdarzenia[j];
                        _aktualneZdarzenia[j] = tymczasowe;
                    }
        }
    }
}