
export const simpleTokenizer = string => string
    .toLowerCase()
    .replace(/[^\w\d]/g, ' ')
    .split(' ')
    .filter(word => word.length > 3)
    // W ten sposób zapewniamy "unikalność" słów; rozwiązanie to przyczynia
    // się jedynie do wzrostu dokładności o 0,5%, dlatego jeśli to 
    // wydajność działania algorytmu jest kluczowa, można z niego zrezygnować.
    // Uwaga: Nie zachowuje koleności wystąpień słów w tekście. Aby ją 
    // zachować, konieczne jest wywoływanie metody .reverse() przed oraz 
    // po filtrowaniu.
    .filter((word, index, arr) => arr.indexOf(word, index+1) === -1)
;

// 3% UTRATY dokładności dla zbioru anych IMDB w porównaniu 
// z funkcją simpleTokenizer
export const bigramTokenizer = string => {
    const unigrams = simpleTokenizer(string);
    const bigrams = [];
    for (let i = 0, len = unigrams.length; i < len - 1; i++) {
        bigrams.push(unigrams[i] + " " + unigrams[i+1]);
    }
    return bigrams;
}

class BayesClassifier {

    constructor(tokenizer = null) {
        this.database = {
            labels: {},
            tokens: {}
        };

        this.tokenizer = (tokenizer !== null) ? tokenizer : simpleTokenizer;

    }

    /**
     * Przeprowadza uczenie na podstawie dokumentu i etykiety.
     * @param label
     * @param text
     */
    train(label, text) {
        this.incrementLabelDocumentCount(label);
        this.tokenizer(text).forEach(token => this.incrementTokenCount(token, label));
    }

    /**
     * Inkrementuje liczbę dokumentów w danej kategorii / etykiecie.
     * @param label
     */
    incrementLabelDocumentCount(label) {
        this.database.labels[label] = this.getLabelDocumentCount(label) + 1;
    }

    /**
     * Zwraca liczbę dokumentów zarejestrowanych dla danej kategorii / etykiety.
     * Jeśli parametr label przyjmie wartość null, zwracana jest całkowita liczba 
     * dokumentów użytych do uczenia.
     * @param label
     */
    getLabelDocumentCount(label = null) {
        if (label) {
            return this.database.labels[label] || 0;
        } else {
            return Object.values(this.database.labels)
                .reduce((sum, count) => sum + count, 0);
        }
    }

    /** 
     * Inkrementuje liczbę wystąpień tokenu dla danej etykiety.
     * @param token
     * @param label
     */
    incrementTokenCount(token, label) {
        if (typeof this.database.tokens[token] === 'undefined') {
            this.database.tokens[token] = {};
        }

        this.database.tokens[token][label] = this.getTokenCount(token, label) + 1;
    }

    /**
     * Zwraca liczbę wystąpień przekazanego tokenu dla konkretnej kategorii / etykiety.
     * Jeśli żadna etykieta nie została przekazana, metoda zwraca całkowitą 
     * liczbę wystąpień tokenu w całym zbiorze uczącym.
     * @param token
     * @param label
     * @returns {*}
     */
    getTokenCount(token, label = null) {
        if (label) {
            return (this.database.tokens[token] || {})[label] || 0;
        } else {
            return Object.values(this.database.tokens[token] || {})
                .reduce((sum, count) => sum + count, 0);
        }
    }


    /**
     * Zwraca wszystkie etykiety zgromadzone podczas procesu uczenia.
     * @returns {Array}
     */
    getAllLabels() {
        return Object.keys(this.database.labels);
    }

    /**
     * Na podstawie przekazanych tokenu i etykiety, metoda oblicza prawdopodobieństwo
     * tego, że dokument będzie mieć podaną etykietę zakładając, że wystąpi w nim podany 
     * token. Do tego celu używamy znacznie prostszego odpowiednika klasyfikatora 
     * bayesowskiego: wyliczmy prawdopodobieństwo wystąpienia tokenu pod warunkiem,
     * że chodzi o podaną kategorię (częstotliwość występowania słów w kategorii).
     * Ta metoda także odpowiednio traktuje tokeny występujące sporadycznie.
     * @param token
     * @param label
     * @returns {liczba}
     */
    calculateTokenScore(token, label) {
        const rareTokenWeight = 3;

        const totalDocumentCount = this.getLabelDocumentCount();
        const labelDocumentCount = this.getLabelDocumentCount(label);
        const notLabelDocumentCount = totalDocumentCount - labelDocumentCount;

        // Zakładając, że równe prawdopodobieństwa dają 1% dokładności, używamy częstotliwości
        // poszczególnych etykiet
        const probLabel = 1 / this.getAllLabels().length;
        const probNotLabel = 1 - probLabel;

        const tokenLabelCount = this.getTokenCount(token, label);
        const tokenTotalCount = this.getTokenCount(token);
        const tokenNotLabelCount = tokenTotalCount - tokenLabelCount;

        const probTokenGivenLabel = tokenLabelCount / labelDocumentCount;
        const probTokenGivenNotLabel = tokenNotLabelCount / notLabelDocumentCount;
        const probTokenLabelSupport = probTokenGivenLabel * probLabel;
        const probTokenNotLabelSupport = probTokenGivenNotLabel * probNotLabel;

        const rawWordScore =
            (probTokenLabelSupport)
            /
            (probTokenLabelSupport + probTokenNotLabelSupport);

        // Uwzględnienie rzadko występujących tokenów: w zasadzie wyznaczamy 
        // średnią ważoną. Zastosujemy to skrócone nazwy zmiennych, aby 
        // uprościć obliczenia:
        // s to "siła" lub "waga"
        // n to sumaryczna liczba wystąpień danego tokenu
        const s = rareTokenWeight;
        const n = tokenTotalCount;
        const adjustedTokenScore =
            ( (s * probLabel) + (n * (rawWordScore || probLabel)) )
            /
            ( s + n );

        return adjustedTokenScore;
    }

    /**
     * Na podstawie strumienia tokenów (dokumentu podzielonego na tokeny), 
     * metoda wylicza prawdopodobieństwo, że dany dokument na podaną etykietę.
     * @param label
     * @param tokens
     * @returns {liczba}
     */
    calculateLabelProbability(label, tokens) {

        // Z góry zakładamy, że prawdopodobieństwa wszystkich etykiet są równe.
        // Alternatywnie można by wyliczyć prawdopodobieństwo na podstawie częstotliwości
        // występowania poszczególnych etykiet.
        const probLabel = 1 / this.getAllLabels().length;

        // Jakie znaczenie musi mieć token, by został uwzględniony?
        // Ich wynik musi być większy od epsilon z domyślnego wyniku tokenu.
        // To rozwiązanie pozwala odrzucić z rozważań nieinteresujące tokeny.
        // To rozwiązanie powoduje wzrost dokładności z 78% do 87,8% (dla e=.17)
        const epsilon = 0.15;
        // Dla każdego tokenu musimy obliczyć "wynik tokenu", czyli prawdopodobieństwo, 
        // że dany dokument należy do kategorii, zakładając że występuje w nim dany token.
        const tokenScores = tokens
            .map(token => this.calculateTokenScore(token, label))
            .filter(score => Math.abs(probLabel - score) > epsilon);

        // Aby uniknąć niedomiaru zmiennoprzecinkowego podczas operowania na naprawdę 
        // małych liczbach, sumujemy prawdopodobieństwa poszczególnych tokenów 
        // w przestrzeni logarytmicznej. To rozwiązanie stosujemy wyłącznie ze względu
        // na operacje zmiennoprzecinkowe, nie powinno ono mieć wpływu na ogólne działanie algorytmu.
        const logSum = tokenScores.reduce((sum, score) => sum + (Math.log(1-score) - Math.log(score)), 0);
        const probability = 1 / (1 + Math.exp(logSum));

        return probability;
    }

    /**
     * Dla przekazanego dokumentu metoda oblicza jego prawdopodobieństwo dla każdej 
     * z etykiet / kategorii występujących w zbiorze uczącym.
     * Pierwszym elementem zwracanej tablicy wynikowej (elementem o indeksie 0), 
     * jest etykieta / kategoria najlepiej pasująca do danego dokumentu.
     * @param text
     * @returns {Array.<Object>}
     */
    calculateAllLabelProbabilities(text) {
        const tokens = this.tokenizer(text);
        return this.getAllLabels()
            .map(label => ({
                label,
                probability: this.calculateLabelProbability(label, tokens)
            }))
            .sort((a, b) => a.probability > b.probability ? -1 : 1);
    }

    /**
     * Dla przekazanego dokumentu metoda przewiduje kategorię do której on należy.
     * @param text
     * @returns {{label: łańcuch, probability: liczba, probabilities: Array}}
     */
    predict(text) {
        const probabilities = this.calculateAllLabelProbabilities(text);
        const best = probabilities[0];

        return {
            label: best.label,
            probability: best.probability,
            probabilities
        };
    }

}

export default BayesClassifier;