/**
 * Wylicza średnią dla tablicy liczb.
 * @param {tablica.<liczba>} liczby
 * @return {liczba}
 */
const mean = numbers => numbers.reduce((sum, val) => sum + val, 0) / numbers.length;

/**
 * Wylicza odległość pomiędzy dwoma punktami.
 * Punkty muszą być zdefiniowane w formie tablicy lub obiektów o odpowiednich kluczach.
 * @param {tablica.<liczba>} a
 * @param {tablica.<liczba>} b
 * @return {liczba}
 */
const distance = (a, b) => Math.sqrt(
    a.map((aPoint, i) => b[i] - aPoint)
    .reduce((sumOfSquares, diff) => sumOfSquares + (diff*diff), 0)
);

/**
 * ! TODO: przetłumaczyć !!
 * Określa centroidy "k" klastrów na podstawie listy punktów,
 * używając do tego celu algorytmu k-średnich.
 *
 * Zastosowanie:
 *
 *     const numberOfCentroids = 3;
 *     const arrayOfDataPoints = [ [1, 2], [3, 3], ... ];
 *     const maxAllowedIterations = 1000;
 *     const kMeansSolver = new KMeans(numberOfCentroids, arrayOfDataPoints);
 *     const {centroids, iteration, error, didReachSteadyState} = kMeansSolver.solve(maxAllowedIterations);
 *     // Reset the solver if you wish to run it again:
 *     kMeansSolver.reset();
 *     // Inspect iteration logs to debug:
 *     console.log(kMeansSolver.iterationLogs)
 *
 * KONIECZNIE należy zwrócić uwagę, by wszystkie punkty danych przekazywane
 * do algorytmu miały taką samą liczbę współrzędnych; na przykład: nie można
 * mieszać punktów dwu- i trzywymiarowych.
 */
class KMeans {

    /**
     * @param k
     * @param data
     */
    constructor(k, data) {
        this.k = k;
        this.data = data;
        this.reset();
    }

    /**
     * Przywraca początkowy stan instancji; należy jej używać kiedy
     * chcemy ponownie użyć tej samej instancji do przetworzenia
     * tego samego zestawu punktów danych, lecz z wykorzystaniem
     * innych warunków początkowych.
     */
    reset() {
        this.error = null;
        this.iterations = 0;
        this.iterationLogs = [];
        this.centroids = this.initRandomCentroids();
        this.centroidAssignments = [];
    }

    /**
     * Określa liczbę wymiarów w zbiorze danych.
     * @return {liczba}
     */
    getDimensionality() {
        const point = this.data[0];
        return point.length;
    }

    /**
     * Dla danego wymiaru w zbiorze danych, metoda określa wartość 
     * minimalną i maksymalną. Jest ona używana podczas losowej 
     * inicjalizacji, by upewnić się, że centroidy będą mieć ten 
     * sam zakres wartości do dane.
     *
     * @param n
     * @returns {{min: *, max: *}}
     */
    getRangeForDimension(n) {
        const values = this.data.map(point => point[n]);
        return {
            min: Math.min.apply(null, values),
            max: Math.max.apply(null, values)
        };
    }

    /**
     * Metoda zwraca zakresy dla wszystkich wymiarów.
     * @see getRangeForDimension
     * @returns {tablica} Tablica, której indeksami są numery wymiarów, a wartościami - wyniki 
     *                    zwrócone przez metodę getRangeForDimension.
     */
    getAllDimensionRanges() {
        const dimensionRanges = [];
        const dimensionality = this.getDimensionality();

        for (let dimension = 0; dimension < dimensionality; dimension++) {
            dimensionRanges[dimension] = this.getRangeForDimension(dimension);
        }

        return dimensionRanges;
    }

    /**
     * Metoda inicjuje losowe centroidy, używając przy tym zakresów danych
     * do określenia wartości minimalnych i maksymalnych dla generowanych 
     * centroidów. 
     * Można wyświetlić wyniki zwracane przez tę metodę, by sprawdzić 
     * działanie inicjalizacji, gdyż normalnie metoda ta jest używana
     * wewnętrznie.
     * @see getAllDimensionRanges
     * @see getRangeForDimension
     * @returns {tablica}
     */
    initRandomCentroids() {

        const dimensionality = this.getDimensionality();
        const dimensionRanges = this.getAllDimensionRanges();
        const centroids = [];

        // Musimy wygenerować 'k' centroidów.
        for (let i = 0; i < this.k; i++) {

            // Ponieważ każdy wymiar ma swój unikalny zakres wartości, dlatego 
            // najpierw tworzymy zmienną pomocniczą.
            let point = [];

            /**
             * Dla każdego wymiaru w danych określamy jego zakres wartości (min/max),
             * a następnie wybieramy losową wartość mieszczącą się w tym zakresie.
             */
            for (let dimension = 0; dimension < dimensionality; dimension++) {
                const {min, max} = dimensionRanges[dimension];
                point[dimension] = min + (Math.random()*(max-min));
            }

            centroids.push(point);

        }

        return centroids;
    }

/**
 * Dysponując punktem wybranym z danych, metoda określa centroid
 * położony najbliżej niego i kojarzy ten punkt znalezionym centroidem.
 * Metoda zwraca wartość logiczną określającą czy skojarzenie punktu
 * z centroidem uległo zmianie czy nie. Wynik ten jest używany do 
 * określenia warunku zakończenia działania algorytmu.
 * @param pointIndex
 * @returns {logiczna} Czy przypisanie punktu do centroidu zmieniło się?
 */
    assignPointToCentroid(pointIndex) {

        const lastAssignedCentroid = this.centroidAssignments[pointIndex];
        const point = this.data[pointIndex];
        let minDistance = null;
        let assignedCentroid = null;

        for (let i = 0; i < this.centroids.length; i++) {
            const centroid = this.centroids[i];
            const distanceToCentroid = distance(point, centroid);

            if (minDistance === null || distanceToCentroid < minDistance) {
                minDistance = distanceToCentroid;
                assignedCentroid = i;
            }

        }

        this.centroidAssignments[pointIndex] = assignedCentroid;

        return lastAssignedCentroid !== assignedCentroid;
    }

    /**
     * Dla wszystkich punktów danych, metoda wywołuje metodę
     * assignPointsToCentroids, po czym zwraca informację, czy 
     * dla _któregokolwiek_ z punktów, jego skojarzenie z centroidem
     * uletło zmianie.
     *
     * @see assignPointToCentroid
     * @returns {logiczna} Czy dla któregokolwiek z punktów skojarzenie centroidem uległo zmianie?
     */
    assignPointsToCentroids() {
        let didAnyPointsGetReassigned = false;
        for (let i = 0; i < this.data.length; i++) {
            const wasReassigned = this.assignPointToCentroid(i);
            if (wasReassigned) didAnyPointsGetReassigned = true;
        }
        return didAnyPointsGetReassigned;
    }

    /**
     * Dla przekazanego centroidu metoda zwraca tablicę wszystkich 
     * skojarzonych z nim punktów.
     *
     * @param centroidIndex
     * @returns {tablica}
     */
    getPointsForCentroid(centroidIndex) {
        const points = [];
        for (let i = 0; i < this.data.length; i++) {
            const assignment = this.centroidAssignments[i];
            if (assignment === centroidIndex) {
                points.push(this.data[i]);
            }
        }
        return points;
    }

    /**
     * Dla przekazanego centroidu metoda wyznacza wartość średnią
     * położenia skojarzonych z nim punktów, a następnie używa jej 
     * jako nowego położenia centroidu.
     * @see getPointsForCentroid
     * @param centroidIndex
     * @returns {tablica}
     */
    updateCentroidLocation(centroidIndex) {
        const thisCentroidPoints = this.getPointsForCentroid(centroidIndex);
        const dimensionality = this.getDimensionality();
        const newCentroid = [];
        for (let dimension = 0; dimension < dimensionality; dimension++) {
            newCentroid[dimension] = mean(thisCentroidPoints.map(point => point[dimension]));
        }
        this.centroids[centroidIndex] = newCentroid;
        return newCentroid;
    }

    /**
     * Dla wszystkich centroidów, metoda wywołuje metodę updateCentroidLocation.
     */
    updateCentroidLocations() {
        for (let i = 0; i < this.centroids.length; i++) {
            this.updateCentroidLocation(i);
        }
    }

    /**
     * Metoda oblicza całkowity "błąd" dla aktualnego stanu centroidu oraz skojarzonych
     * z nim punktów danych. 
     * W tym przypadku błąd jest zdefiniowanych jako średniokwardatowa odległość 
     * wszystkich punktów od centroidu z którym są one skojarzone.
     * @returns {Number}
     */
    calculateError() {

        let sumDistanceSquared = 0;
        for (let i = 0; i < this.data.length; i++) {
            const centroidIndex = this.centroidAssignments[i];
            const centroid = this.centroids[centroidIndex];
            const point = this.data[i];

            // Komentarz z tego wiersza można usunąć jeśli chcemy, by byłąd
            // był wyliczany w całkowicie geometryczny sposób.
            // const thisDistance = distance(point, centroid);
            
            // Ta wersja uwzględnia także liczbę klastrów, przydatna dla 
            // obiektu działającego automatycznie, by unikać nadmiernego dopasowania.
            const thisDistance = distance(point, centroid) + this.k;

            sumDistanceSquared += thisDistance*thisDistance;
        }

        this.error = Math.sqrt(sumDistanceSquared / this.data.length);
        return this.error;
    }

    /**
     * Metoda wykonuje algorytm k-średnich tak długo aż osiągnie on stan stabilności
     * lub zostanie przekroczona wartość maxIterations.
     *
     * Wartością zwracaną przez tę metodę jest obiekt o następujących właściwościach:
     * {
     *  centroids {Array.<Object>},
     *  iteration {number},
     *  error {number},
     *  didReachSteadyState {Boolean}
     * }
     *
     * Najprawdopodobniej nas będzie najbardziej interesować właściwość centroids
     * tego obiektu.
     *
     * @param {liczba} maxIterations wartością domyślną jest 1000
     * @returns {obiekt}
     */
    solve(maxIterations = 1000) {

        while (this.iterations < maxIterations) {

            const didAssignmentsChange = this.assignPointsToCentroids();
            this.updateCentroidLocations();
            this.calculateError();


            this.iterationLogs[this.iterations] = {
                centroids: [...this.centroids],
                iteration: this.iterations,
                error: this.error,
                didReachSteadyState: !didAssignmentsChange
            };

            if (didAssignmentsChange === false) {
                break;
            }

            this.iterations++;

        }

        return this.iterationLogs[this.iterationLogs.length - 1];
    }

}


export default KMeans;