
// Dla wygody deklarujemy zmienne dla wszystkich często wykorzystywanych obiektów
var b2Vec2 = Box2D.Common.Math.b2Vec2;
var b2BodyDef = Box2D.Dynamics.b2BodyDef;
var b2Body = Box2D.Dynamics.b2Body;
var b2FixtureDef = Box2D.Dynamics.b2FixtureDef;
var b2World = Box2D.Dynamics.b2World;
var b2PolygonShape = Box2D.Collision.Shapes.b2PolygonShape;
var b2CircleShape = Box2D.Collision.Shapes.b2CircleShape;
var b2DebugDraw = Box2D.Dynamics.b2DebugDraw;
var b2ContactListener = Box2D.Dynamics.b2ContactListener;

var game = {
    // Zaczynamy inicjalizowanie obiektów, wczytujemy zasoby i wyświetlamy ekran powitalny
    init: function() {
        // Pobieramy uchwyt elementu canvas oraz kontekstu gry
        game.canvas = document.getElementById("gamecanvas");
        game.context = game.canvas.getContext("2d");

        // Inicjalizujemy obiekty
        levels.init();
        loader.init();
        mouse.init();

        // Wczytujemy wszystkie efekty dźwiękowe i muzykę tła
        game.loadSounds(function() {
            // Ukrywamy wszystkie poziomy gry oraz wyświetlamy ekran startowy
            game.hideScreens();
            game.showScreen("gamestartscreen");
        });


    },

    // Wywoływana po kliknięciu przycisku odtwarzania
    playGame: function() {

        // Inicjalizacja dźwięku w przeglądarce mobilnej Safari
        if (window.wAudio) {
            window.wAudio.playMutedSound();
        }

        game.showLevelScreen();
    },

    scale: 1,
    resize: function() {

        var maxWidth = window.innerWidth;
        var maxHeight = window.innerHeight;

        var scale = Math.min(maxWidth / 640, maxHeight / 480);

        var gameContainer = document.getElementById("gamecontainer");

        gameContainer.style.transform = "translate(-50%, -50%) " + "scale(" + scale + ")";

        // Znajdujemy maksymalną szerokość na podstawie bieżącej skali
        // i wybieramy wartość spomiędzy 640 i 1024
        var width = Math.max(640, Math.min(1024, maxWidth / scale ));

        // Przypisujemy nową szerokość do kontenera gry i obiektu canvas gry
        gameContainer.style.width = width + "px";

        var gameCanvas = document.getElementById("gamecanvas");

        gameCanvas.width = width;

        game.scale = scale;
    },

    loadSounds: function(onload) {
        game.backgroundMusic = loader.loadSound("audio/gurdonark-kindergarten");

        game.slingshotReleasedSound = loader.loadSound("audio/released");
        game.bounceSound = loader.loadSound("audio/bounce");
        game.breakSound = {
            "glass": loader.loadSound("audio/glassbreak"),
            "wood": loader.loadSound("audio/woodbreak")
        };

        loader.onload = onload;
    },

    startBackgroundMusic: function() {
        game.backgroundMusic.play();
        game.setBackgroundMusicButton();
    },

    stopBackgroundMusic: function() {
        game.backgroundMusic.pause();
        // Powracamy na początek utworu
        game.backgroundMusic.currentTime = 0;

        game.setBackgroundMusicButton();
    },

    toggleBackgroundMusic: function() {
        if (game.backgroundMusic.paused) {
            game.backgroundMusic.play();
        } else {
            game.backgroundMusic.pause();
        }

        game.setBackgroundMusicButton();
    },

    setBackgroundMusicButton: function() {
        var toggleImage = document.getElementById("togglemusic");

        if (game.backgroundMusic.paused) {
            toggleImage.src = "images/icons/nosound.png";
        } else {
            toggleImage.src = "images/icons/sound.png";
        }
    },

    hideScreens: function() {
        var screens = document.getElementsByClassName("gamelayer");

        // Iterujemy wszystkie warstwy gry i wyłączamy ich wyświetlanie
        for (let i = screens.length - 1; i >= 0; i--) {
            var screen = screens[i];

            screen.style.display = "none";
        }
    },

    hideScreen: function(id) {
        var screen = document.getElementById(id);

        screen.style.display = "none";
    },

    showScreen: function(id) {
        var screen = document.getElementById(id);

        screen.style.display = "block";
    },

    showLevelScreen: function() {
        game.hideScreens();
        game.showScreen("levelselectscreen");
    },

    restartLevel: function() {
        window.cancelAnimationFrame(game.animationFrame);
        game.lastUpdateTime = undefined;
        levels.load(game.currentLevel.number);
    },

    startNextLevel: function() {
        window.cancelAnimationFrame(game.animationFrame);
        game.lastUpdateTime = undefined;
        levels.load(game.currentLevel.number + 1);
    },

    // Zapisujemy bieżący stan gry: intro, wait-for-firing, firing, fired, load-next-hero, success, failure
    mode: "intro",

    // Współrzędne X i Y procy
    slingshotX: 140,
    slingshotY: 280,

    // Współrzędne X i Y punktu, w którym taśma jest przyczepiona do procy
    slingshotBandX: 140 + 55,
    slingshotBandY: 280 + 23,

    // Flaga służąca do sprawdzenia, czy gra się zakończyła
    ended: false,

    // Wynik gry
    score: 0,

    // Przesunięcie wzdłuż osi X, na podstawie którego przesuwamy ekran od lewej do prawej
    offsetLeft: 0,

    start: function() {
        game.hideScreens();

        // Wyświetlamy element canvas gry oraz wynik
        game.showScreen("gamecanvas");
        game.showScreen("scorescreen");

        game.mode = "intro";
        game.currentHero = undefined;

        game.offsetLeft = 0;
        game.ended = false;

        game.animationFrame = window.requestAnimationFrame(game.animate, game.canvas);

        // Odtwarzamy muzykę w momencie uruchomienia gry
        game.startBackgroundMusic();
    },

    // Maksymalna prędkość przesuwania klatki w pikselach
    maxSpeed: 3,

    // Przesuwamy ekran, aby jego środek znajdował się w punkcie newCenter
    // (a przynajmniej możliwie jak najbliżej)
    panTo: function(newCenter) {

            // Minimalne i maksymalne przesunięcie
        var minOffset = 0;
        var maxOffset = game.currentLevel.backgroundImage.width - game.canvas.width;

        // Bieżący środek ekranu znajduje się w odległości połowy szerokości ekranu od lewego przesunięcia
        var currentCenter = game.offsetLeft + game.canvas.width / 2;

        // Kontynuujemy przesuwanie, jeśli odległość między nowym a bieżącym środkiem jest większa od 0 i nie przekroczyliśmy minimalnego i maksymalnego przesunięcia
        if (Math.abs(newCenter - currentCenter) > 0 && game.offsetLeft <= maxOffset && game.offsetLeft >= minOffset) {
            // W każdym tyknięciu przesuwamy o połowę odległości od punktu newCenter do punktu currentCenter
            // To ułatwia spowalnianie ruchu
            var deltaX = (newCenter - currentCenter) / 2;

            // Jednakże jeśli wartość deltaX jest bardzo wysoka, ekran będzie się przesuwać zbyt szybko, dlatego jeśli jest ona większa niż wartość maxSpeed
            if (Math.abs(deltaX) > game.maxSpeed) {
                // Ograniczamy wartość deltaX do wartości game.maxSpeed (nie zmieniając znaku zmiennej deltaX)
                deltaX = game.maxSpeed * Math.sign(deltaX);
            }

            // Jednakże jeśli wartość deltaX jest bardzo wysoka, ekran będzie się przesuwać zbyt szybko, dlatego jeśli jest ona większa niż wartość maxSpeed
            if (Math.abs(deltaX) <= 1) {
                deltaX = (newCenter - currentCenter);
            }

            // Ograniczamy wartość deltaX do wartości game.maxSpeed (nie zmieniając znaku zmiennej deltaX)
            game.offsetLeft += deltaX;

            // Upewniamy się, że nie przekroczyliśmy minimalnego i maksymalnego limitu
            if (game.offsetLeft <= minOffset) {
                game.offsetLeft = minOffset;

                // Informujemy funkcję wywołującą, że przesunęliśmy się możliwie najbliżej do punktu newCenter
                return true;
            } else if (game.offsetLeft >= maxOffset) {
                game.offsetLeft = maxOffset;

                // Informujemy funkcję wywołującą, że przesunęliśmy się możliwie najbliżej do punktu newCenter
                return true;
            }

        } else {
            // Informujemy funkcję wywołującą, że przesunęliśmy się możliwie najbliżej do punktu newCenter
            return true;
        }
    },

    // Sprawdzamy wszystkich bohaterów i czarne charaktery pozostałe w świecie Box2d i zapisujemy ich ciała Box2D
    heroes: undefined,
    villains: undefined,
    countHeroesAndVillains: function() {
        game.heroes = [];
        game.villains = [];
        for (let body = box2d.world.GetBodyList(); body; body = body.GetNext()) {
            var entity = body.GetUserData();

            if (entity) {
                if (entity.type === "hero") {
                    game.heroes.push(body);
                } else if (entity.type === "villain") {
                    game.villains.push(body);
                }
            }
        }
    },

    handleGameLogic: function() {
        if (game.mode === "intro") {
            if (game.panTo(700)) {
                game.mode = "load-next-hero";
            }
        }


        if (game.mode === "wait-for-firing") {
            if (mouse.dragging) {
                if (game.mouseOnCurrentHero()) {
                    game.mode = "firing";
                } else {
                    game.panTo(mouse.x + game.offsetLeft);
                }
            } else {
                game.panTo(game.slingshotX);
            }
        }

        if (game.mode === "firing") {

            if (mouse.down) {
                game.panTo(game.slingshotX);

                // Ograniczamy przeciąganie do odległości maxDragDistance
                var distance = Math.pow(Math.pow(mouse.x - game.slingshotBandX + game.offsetLeft, 2) + Math.pow(mouse.y - game.slingshotBandY, 2), 0.5);
                var angle = Math.atan2(mouse.y - game.slingshotBandY, mouse.x - game.slingshotBandX);

                var minDragDistance = 10;
                var maxDragDistance = 120;
                var maxAngle = Math.PI * 145 / 180;

                if (angle > 0 && angle < maxAngle ) {
                    angle = maxAngle;
                }
                if (angle < 0 && angle > -maxAngle ) {
                    angle = -maxAngle;
                }
                // Jeśli przesuniemy bohatera za daleko, ograniczamy ruch
                if (distance > maxDragDistance) {
                    distance = maxDragDistance;
                }

                // Jeśli przesuniemy bohatera w niewłaściwym kierunku, ograniczamy ruch
                if ((mouse.x + game.offsetLeft > game.slingshotBandX)) {
                    distance = minDragDistance;
                    angle = Math.PI;
                }

                // Umieszczamy bohatera w pozycji określonej na podstawie odległości i kąta, obliczonych wcześniej
                game.currentHero.SetPosition({ x: (game.slingshotBandX + distance * Math.cos(angle) + game.offsetLeft) / box2d.scale,
                    y: (game.slingshotBandY + distance * Math.sin(angle)) / box2d.scale });

            } else {
                game.mode = "fired";
                var impulseScaleFactor = 0.8;
                var heroPosition = game.currentHero.GetPosition();
                var heroPositionX = heroPosition.x * box2d.scale;
                var heroPositionY = heroPosition.y * box2d.scale;

                var impulse = new b2Vec2((game.slingshotBandX - heroPositionX) * impulseScaleFactor,
                    (game.slingshotBandY - heroPositionY) * impulseScaleFactor);

                // Przykładamy siłę do bohatera, aby go wystrzelić w kierunku celu
                game.currentHero.ApplyImpulse(impulse, game.currentHero.GetWorldCenter());

                // Upewniamy się, że bohater nie będzie się obracać w nieskończoność
                game.currentHero.SetAngularDamping(2);

                // Odtwarzamy dźwięk zwalniania procy
                game.slingshotReleasedSound.play();
            }
        }

        if (game.mode === "fired") {
            // Przesuwamy się w położenie bieżącego lecącego bohatera
            var heroX = game.currentHero.GetPosition().x * box2d.scale;

            game.panTo(heroX);

            // Czekamy, aż bohater przestanie się poruszać lub przekroczy granice
            if (!game.currentHero.IsAwake() || heroX < 0 || heroX > game.currentLevel.foregroundImage.width) {
                // następnie usuwamy bohatera ze świata box2d
                box2d.world.DestroyBody(game.currentHero);
                // czyścimy bieżącego bohatera
                game.currentHero = undefined;
                // i wczytujemy następnego bohatera
                game.mode = "load-next-hero";
            }
        }

        if (game.mode === "load-next-hero") {

            // Najpierw liczymy bohaterów i czarne charaktery i wypełniamy odpowiednie tablice
            game.countHeroesAndVillains();

            // Sprawdzamy, czy jakiś czarny charakter jest żywy, a jeśli nie, kończymy poziom (sukces)
            if (game.villains.length === 0) {
                game.mode = "level-success";

                return;
            }

            // Sprawdzamy, czy istnieje jeszcze jakiś bohater do wczytania, a jeśli nie, kończymy poziom
            if (game.heroes.length === 0) {
                game.mode = "level-failure";

                return;
            }

            // Wczytujemy bohatera i ustawiamy tryb na wait-for-firing
            if (!game.currentHero) {
                // Wybieramy ostatniego bohatera z tablicy bohaterów
                game.currentHero = game.heroes[game.heroes.length - 1];

                // Początkowe położenie, w którym należy wczytać bohatera
                var heroStartX = 180;
                var heroStartY = 180;

                // Umieszczamy go w powietrzu, nieco nad procą
                game.currentHero.SetPosition({ x: heroStartX / box2d.scale, y: heroStartY / box2d.scale });
                game.currentHero.SetLinearVelocity({ x: 0, y: 0 });
                game.currentHero.SetAngularVelocity(0);

                // Ponieważ bohater znajdował się na ziemi i był "uśpiony" w Box2D, musimy go "obudzić"
                game.currentHero.SetAwake(true);
            } else {
                // Czekamy, aż bohater przestanie podskakiwać na procy i przejdzie w stan spoczynku
                // a następnie przełączamy tryb w wait-for-firing
                game.panTo(game.slingshotX);
                if (!game.currentHero.IsAwake()) {
                    game.mode = "wait-for-firing";
                }
            }
        }

        if (game.mode === "level-success" || game.mode === "level-failure") {
            // Najpierw przesuwamy się do końca w lewo
            if (game.panTo(0)) {
                // Następnie pokazujemy, że gra się skończyła i wyświetlamy ekran końcowy
                game.ended = true;
                game.showEndingScreen();
            }
        }
    },

    mouseOnCurrentHero: function() {
        if (!game.currentHero) {
            return false;
        }

        var position = game.currentHero.GetPosition();

        // Odległość między środkiem bohatera, a kursorem myszy
        var distanceSquared = Math.pow(position.x * box2d.scale - mouse.x - game.offsetLeft, 2) +
            Math.pow(position.y * box2d.scale - mouse.y, 2);

        // Promień bohatera
        var radiusSquared = Math.pow(game.currentHero.GetUserData().radius, 2);

        // Jeśli odległość kursora myszy od środka bohatera jest mniejsza od promienia, mysz wskazuje tego bohatera
        return (distanceSquared <= radiusSquared);
    },

    animate: function() {
        // Animowanie postaci
        var currentTime = new Date().getTime();

        if (game.lastUpdateTime) {
            var timeStep = (currentTime - game.lastUpdateTime) / 1000;

            box2d.step(timeStep);

            // Ustawiamy wartość maxSpeed na 3 piksele jeśli wartość timeStep wynosi 1/60
            game.maxSpeed = Math.round(timeStep * 3 * 60);
        }

        game.lastUpdateTime = currentTime;

        // Obsługa przesuwania, stanów gry oraz sterowanie przepływem
        game.handleGameLogic();

        // Usuwamy wszystkie ciała, których życie dobiegło końca podczas tego cyklu animacji
        game.removeDeadBodies();

        // Rysujemy tło i definiujemy przewijanie z efektem paralaksy
        // Najpierw rysujemy obraz tła przesunięty o ułamek odległości offsetLeft (1/4)
        // Im większa wartość ułamka, tym bliższe wydaje się tło
        game.context.drawImage(game.currentLevel.backgroundImage, game.offsetLeft / 4, 0, game.canvas.width, game.canvas.height, 0, 0, game.canvas.width, game.canvas.height);
        // Następnie rysujemy obraz pierwszego planu przesunięty o całą odległość offsetLeft
        game.context.drawImage(game.currentLevel.foregroundImage, game.offsetLeft, 0, game.canvas.width, game.canvas.height, 0, 0, game.canvas.width, game.canvas.height);


        // Rysujemy podstawę procy przesuniętą o całą odległość offsetLeft
        game.context.drawImage(game.slingshotImage, game.slingshotX - game.offsetLeft, game.slingshotY);

        // Rysujemy wszystkie ciała
        game.drawAllBodies();

        // Rysujemy pasek podczas wystrzeliwania bohatera
        if (game.mode === "firing") {
            game.drawSlingshotBand();
        }

        // Rysujemy przód procy przesunięty o całą odległość offsetLeft
        game.context.drawImage(game.slingshotFrontImage, game.slingshotX - game.offsetLeft, game.slingshotY);

        if (!game.ended) {
            game.animationFrame = window.requestAnimationFrame(game.animate, game.canvas);
        }
    },

    drawAllBodies: function() {
        // Rysujemy dane w trybie debugowania, jeśli został skonfigurowany obiekt canvas przeznaczony dla tego trybu
        if (box2d.debugCanvas) {
            box2d.world.DrawDebugData();
        }

        // Iterujemy wszystkie ciała i rysujemy je w obiekcie canvas gry
        for (let body = box2d.world.GetBodyList(); body; body = body.GetNext()) {
            var entity = body.GetUserData();


            if (entity) {
                entities.draw(entity, body.GetPosition(), body.GetAngle());
            }
        }

    },

    drawSlingshotBand: function() {
        game.context.strokeStyle = "rgb(68,31,11)"; // Dark brown color
        game.context.lineWidth = 7; // Draw a thick line

        // Na podstawie kąta i odległości odciągnięcia bohatera obliczamy współrzędne krawędzi bohatera względem jego środka
        var radius = game.currentHero.GetUserData().radius + 1; // 1 piksel dodatkowego marginesu
        var heroX = game.currentHero.GetPosition().x * box2d.scale;
        var heroY = game.currentHero.GetPosition().y * box2d.scale;
        var angle = Math.atan2(game.slingshotBandY - heroY, game.slingshotBandX - heroX);

        // To współrzędne X i Y punktu, w którym taśma styka się z bohaterem
        var heroFarEdgeX = heroX - radius * Math.cos(angle);
        var heroFarEdgeY = heroY - radius * Math.sin(angle);


        game.context.beginPath();
        // Zaczynamy rysowanie linii od wierzchołka procy (z tyłu)
        game.context.moveTo(game.slingshotBandX - game.offsetLeft, game.slingshotBandY);

        // Rysujemy linię do środka bohatera
        game.context.lineTo(heroX - game.offsetLeft, heroY);
        game.context.stroke();

        // Rysujemy bohatera nad tylną taśmą
        entities.draw(game.currentHero.GetUserData(), game.currentHero.GetPosition(), game.currentHero.GetAngle());

        game.context.beginPath();
        // Przesuwamy się do krawędzi bohatera najbardziej oddalonej od wierzchołka procy
        game.context.moveTo(heroFarEdgeX - game.offsetLeft, heroFarEdgeY);

        // Rysujemy linię z powrotem do wierzchołka procy (z przodu)
        game.context.lineTo(game.slingshotBandX - game.offsetLeft - 40, game.slingshotBandY + 15);
        game.context.stroke();
    },

    removeDeadBodies: function() {

        // Iterujemy wszystkie ciała
        for (let body = box2d.world.GetBodyList(); body; body = body.GetNext()) {
            var entity = body.GetUserData();

            if (entity) {
                var entityX = body.GetPosition().x * box2d.scale;

                // Jeśli ciało znajdzie się poza granicami, lub wartość właściwości health jest mniejsza od 0
                if (entityX < 0 || entityX > game.currentLevel.foregroundImage.width ||
                    (entity.health !== undefined && entity.health <= 0)) {

                    // Usuwamy encję ze świata box2d
                    box2d.world.DestroyBody(body);

                    // Uaktualniamy punktację, jeśli czarny charakter zostanie zabity
                    if (entity.type === "villain") {
                        game.score += entity.calories;
                        document.getElementById("score").innerHTML = "Wynik: " + game.score;
                    }

                    // Jeśli do encji przydzielony jest dźwięk rozbijania, odtwarzamy go
                    if (entity.breakSound) {
                        entity.breakSound.play();
                    }
                }
            }
        }
    },

    showEndingScreen: function() {
        var playNextLevel = document.getElementById("playnextlevel");
        var endingMessage = document.getElementById("endingmessage");

        if (game.mode === "level-success") {
            if (game.currentLevel.number < levels.data.length - 1) {
                endingMessage.innerHTML = "Poziom zakończony. Świetna robota!!!";
                // Dostępnych jest więcej poziomów. Pokazujemy przycisk następnego poziomu
                playNextLevel.style.display = "block";
            } else {
                endingMessage.innerHTML = "Wszystkie poziomy zostały zakończone. Świetna robota!!!";
                // Brak kolejnych poziomów. Ukrywamy przycisk następnego poziomu
                playNextLevel.style.display = "none";
            }
        } else if (game.mode === "level-failure") {
            endingMessage.innerHTML = "Przegrałeś. Chcesz zagrać ponownie?";
            // Poziom zakończony niepowodzeniem. Ukrywamy przycisk kolejnego poziomu
            playNextLevel.style.display = "none";
        }

        game.showScreen("endingscreen");

        // W momencie zakończenia gry zatrzymujemy odtwarzanie muzyki w tle
        game.stopBackgroundMusic();
    },

    step: function(timeStep) {
        // Zgodnie z dokumentacją Box2D, jeśli wartość timeStep jest większa niż 1 / 30,
        // Box2D może mieć problemy z wykrywaniem kolizji
        // Dlatego przycinamy timeStep do 1 / 30

        if (timeStep > 1 / 30) {
            timeStep = 1 / 30;
        }

        // velocity iterations = 8
        // position iterations = 3

        box2d.world.Step(timeStep, 8, 3);
    }
};

var levels = {
    // Dane poziomu
    data: [{   // Pierwszy poziom
        foreground: "desert-foreground",
        background: "clouds-background",
        entities: [
            // Podłoże
            { type: "ground", name: "dirt", x: 500, y: 440, width: 1000, height: 20, isStatic: true },
            // Drewniana rama procy
            { type: "ground", name: "wood", x: 190, y: 390, width: 30, height: 80, isStatic: true },

            { type: "block", name: "wood", x: 500, y: 380, angle: 90, width: 100, height: 25 },
            { type: "block", name: "glass", x: 500, y: 280, angle: 90, width: 100, height: 25 },
            { type: "villain", name: "burger", x: 500, y: 205, calories: 590 },

            { type: "block", name: "wood", x: 800, y: 380, angle: 90, width: 100, height: 25 },
            { type: "block", name: "glass", x: 800, y: 280, angle: 90, width: 100, height: 25 },
            { type: "villain", name: "fries", x: 800, y: 205, calories: 420 },

            { type: "hero", name: "orange", x: 80, y: 405 },
            { type: "hero", name: "apple", x: 140, y: 405 }
        ]
    }, {   // Drugi poziom
        foreground: "desert-foreground",
        background: "clouds-background",
        entities: [
            // Podłoże
            { type: "ground", name: "dirt", x: 500, y: 440, width: 1000, height: 20, isStatic: true },
            // Drewniana rama procy
            { type: "ground", name: "wood", x: 190, y: 390, width: 30, height: 80, isStatic: true },

            { type: "block", name: "wood", x: 850, y: 380, angle: 90, width: 100, height: 25 },
            { type: "block", name: "wood", x: 700, y: 380, angle: 90, width: 100, height: 25 },
            { type: "block", name: "wood", x: 550, y: 380, angle: 90, width: 100, height: 25 },
            { type: "block", name: "glass", x: 625, y: 316, width: 150, height: 25 },
            { type: "block", name: "glass", x: 775, y: 316, width: 150, height: 25 },

            { type: "block", name: "glass", x: 625, y: 252, angle: 90, width: 100, height: 25 },
            { type: "block", name: "glass", x: 775, y: 252, angle: 90, width: 100, height: 25 },
            { type: "block", name: "wood", x: 700, y: 190, width: 150, height: 25 },

            { type: "villain", name: "burger", x: 700, y: 152, calories: 590 },
            { type: "villain", name: "fries", x: 625, y: 405, calories: 420 },
            { type: "villain", name: "sodacan", x: 775, y: 400, calories: 150 },

            { type: "hero", name: "strawberry", x: 30, y: 415 },
            { type: "hero", name: "orange", x: 80, y: 405 },
            { type: "hero", name: "apple", x: 140, y: 405 }
        ]
    }],

    // Inicjalizowanie ekranu wyboru poziomów
    init: function() {
        var levelSelectScreen = document.getElementById("levelselectscreen");

        // Funkcja obsługi zdarzeń, którą należy wywołać
        var buttonClickHandler = function() {
            game.hideScreen("levelselectscreen");

            // Etykiety poziomów mają wartości 1 i 2. Poziomy mają wartości 0 i 1.
            levels.load(this.value - 1);
        };


        for (let i = 0; i < levels.data.length; i++) {
            var button = document.createElement("input");

            button.type = "button";
            button.value = (i + 1); // Etykiety poziomów mają wartości 1 i 2
            button.addEventListener("click", buttonClickHandler);

            levelSelectScreen.appendChild(button);
        }

    },

    // Wczytujemy wszystkie dane i obrazy dla określonego poziomu
    load: function(number) {

        // Inicjalizujemy świat Box2D po wczytaniu poziomu
        box2d.init();

        // Deklarujemy nowy obiekt currentLevel
        game.currentLevel = { number: number, hero: [] };
        game.score = 0;

        document.getElementById("score").innerHTML = "Wynik: " + game.score;
        var level = levels.data[number];

        // Wczytujemy obrazy tła, pierwszego planu i procy
        game.currentLevel.backgroundImage = loader.loadImage("images/backgrounds/" + level.background + ".png");
        game.currentLevel.foregroundImage = loader.loadImage("images/backgrounds/" + level.foreground + ".png");
        game.slingshotImage = loader.loadImage("images/slingshot.png");
        game.slingshotFrontImage = loader.loadImage("images/slingshot-front.png");

        // Wczytujemy wszystkie encje
        for (let i = level.entities.length - 1; i >= 0; i--) {
            var entity = level.entities[i];

            entities.create(entity);
        }

        // Po wczytaniu zasobów wywołujemy metodę game.start()
        loader.onload = game.start;

    }
};

var loader = {
    loaded: true,
    loadedCount: 0, // Zasoby, które zostały dotychczas wczytane
    totalCount: 0, // Całkowita liczba zasobów, które oczekują na wczytanie

    init: function() {
        // Sprawdzamy wsparcie dla zasobów dźwiękowych
        var mp3Support, oggSupport;
        var audio = document.createElement("audio");

        if (audio.canPlayType) {
               // Obecnie metoda canPlayType() zwraca: "", "maybe" lub "probably"
            mp3Support = "" !== audio.canPlayType("audio/mpeg");
            oggSupport = "" !== audio.canPlayType("audio/ogg; codecs=\"vorbis\"");
        } else {
            // Brak wsparcia dla znacznika audio
            mp3Support = false;
            oggSupport = false;
        }

        // Sprawdzamy wsparcie formatu ogg, następnie mp3, a w razie niepowodzenia ustawiamy wartość zmiennej soundFileExtn na undefined
        loader.soundFileExtn = oggSupport ? ".ogg" : mp3Support ? ".mp3" : undefined;
    },

    loadImage: function(url) {
        this.loaded = false;
        this.totalCount++;

        game.showScreen("loadingscreen");

        var image = new Image();

        image.addEventListener("load", loader.itemLoaded, false);
        image.src = url;

        return image;
    },

    soundFileExtn: ".ogg",

    loadSound: function(url) {
        this.loaded = false;
        this.totalCount++;

        game.showScreen("loadingscreen");

        var audio = new (window.wAudio || Audio)();

        audio.addEventListener("canplaythrough", loader.itemLoaded, false);
        audio.src = url + loader.soundFileExtn;

        return audio;
    },

    itemLoaded: function(ev) {
        // Przestajemy oczekiwać na typ zdarzenia (load lub canplaythrough) dla elementu, który został już wczytany
        ev.target.removeEventListener(ev.type, loader.itemLoaded, false);

        loader.loadedCount++;

        document.getElementById("loadingmessage").innerHTML = "Loaded " + loader.loadedCount + " of " + loader.totalCount;

        if (loader.loadedCount === loader.totalCount) {
            // Wczytywanie zostało zakończone
            // Resetujemy i czyścimy obiekt wczytujący
            loader.loaded = true;
            loader.loadedCount = 0;
            loader.totalCount = 0;

            // Ukrywamy ekran wczytywania
            game.hideScreen("loadingscreen");

            // i wywołujemy metodę loader.onload, o ile istnieje
            if (loader.onload) {
                loader.onload();
                loader.onload = undefined;
            }
        }
    }
};

var mouse = {
    x: 0,
    y: 0,
    down: false,
    dragging: false,

    init: function() {
        var canvas = document.getElementById("gamecanvas");

        canvas.addEventListener("mousemove", mouse.mousemovehandler, false);
        canvas.addEventListener("mousedown", mouse.mousedownhandler, false);
        canvas.addEventListener("mouseup", mouse.mouseuphandler, false);
        canvas.addEventListener("mouseout", mouse.mouseuphandler, false);

        // Osobno obsługujemy zdarzenie touchmove
        canvas.addEventListener("touchmove", mouse.touchmovehandler, false);

        // Ponownie wykorzystujemy obsługę zdarzeń myszy dla zdarzeń touchstart, touchend, touchcancel
        canvas.addEventListener("touchstart", mouse.mousedownhandler, false);
        canvas.addEventListener("touchend", mouse.mouseuphandler, false);
        canvas.addEventListener("touchcancel", mouse.mouseuphandler, false);
    },

    mousemovehandler: function(ev) {
        var offset = game.canvas.getBoundingClientRect();

        mouse.x = (ev.clientX - offset.left) / game.scale;
        mouse.y = (ev.clientY - offset.top) / game.scale;

        if (mouse.down) {
            mouse.dragging = true;
        }

        ev.preventDefault();
    },

    touchmovehandler: function(ev) {
        var touch = ev.targetTouches[0];
        var offset = game.canvas.getBoundingClientRect();

        mouse.x = (touch.clientX - offset.left) / game.scale;
        mouse.y = (touch.clientY - offset.top) / game.scale;

        if (mouse.down) {
            mouse.dragging = true;
        }

        ev.preventDefault();
    },

    mousedownhandler: function(ev) {
        mouse.down = true;

        ev.preventDefault();
    },

    mouseuphandler: function(ev) {
        mouse.down = false;
        mouse.dragging = false;

        ev.preventDefault();
    }
};

var entities = {
    definitions: {
        "glass": {
            fullHealth: 100,
            density: 2.4,
            friction: 0.4,
            restitution: 0.15
        },
        "wood": {
            fullHealth: 500,
            density: 0.7,
            friction: 0.4,
            restitution: 0.4
        },
        "dirt": {
            density: 3.0,
            friction: 1.5,
            restitution: 0.2
        },
        "burger": {
            shape: "circle",
            fullHealth: 40,
            radius: 25,
            density: 1,
            friction: 0.5,
            restitution: 0.4
        },
        "sodacan": {
            shape: "rectangle",
            fullHealth: 80,
            width: 40,
            height: 60,
            density: 1,
            friction: 0.5,
            restitution: 0.7
        },
        "fries": {
            shape: "rectangle",
            fullHealth: 50,
            width: 40,
            height: 50,
            density: 1,
            friction: 0.5,
            restitution: 0.6
        },
        "apple": {
            shape: "circle",
            radius: 25,
            density: 1.5,
            friction: 0.5,
            restitution: 0.4
        },
        "orange": {
            shape: "circle",
            radius: 25,
            density: 1.5,
            friction: 0.5,
            restitution: 0.4
        },
        "strawberry": {
            shape: "circle",
            radius: 15,
            density: 2.0,
            friction: 0.5,
            restitution: 0.4
        }
    },

    // Pobieramy encję, tworzymy ciało Box2D i dodajemy je do świata
    create: function(entity) {
        var definition = entities.definitions[entity.name];

        if (!definition) {
            console.log("Undefined entity name", entity.name);

            return;
        }

        switch(entity.type) {
            case "block": // proste prostokąty
                entity.health = definition.fullHealth;
                entity.fullHealth = definition.fullHealth;
                entity.shape = "rectangle";
                entity.sprite = loader.loadImage("images/entities/" + entity.name + ".png");

                entity.breakSound = game.breakSound[entity.name];

                box2d.createRectangle(entity, definition);
                break;
            case "ground": // proste prostokąty
                    // Nie musimy definiować kondycji. Te obiekty nie są zniszczalne
                entity.shape = "rectangle";
                    // Nie potrzebujemy sprite'ów. Nie zostaną wcale narysowane
                box2d.createRectangle(entity, definition);
                break;
            case "hero":    // proste koła
            case "villain": // mogą to być koła lub prostokąty
                entity.health = definition.fullHealth;
                entity.fullHealth = definition.fullHealth;
                entity.sprite = loader.loadImage("images/entities/" + entity.name + ".png");
                entity.shape = definition.shape;

                entity.bounceSound = game.bounceSound;

                if (definition.shape === "circle") {
                    entity.radius = definition.radius;
                    box2d.createCircle(entity, definition);
                } else if (definition.shape === "rectangle") {
                    entity.width = definition.width;
                    entity.height = definition.height;
                    box2d.createRectangle(entity, definition);
                }
                break;
            default:
                console.log("Undefined entity type", entity.type);
                break;
        }
    },

    // Pobieramy encję, jej położenie i kąt, a następnie rysujemy ją w obiekcie canvas gry
    draw: function(entity, position, angle) {

        game.context.translate(position.x * box2d.scale - game.offsetLeft, position.y * box2d.scale);
        game.context.rotate(angle);
        var padding = 1;

        switch (entity.type) {
            case "block":
                game.context.drawImage(entity.sprite, 0, 0, entity.sprite.width, entity.sprite.height,
                            -entity.width / 2 - padding, -entity.height / 2 - padding, entity.width + 2 * padding, entity.height + 2 * padding);
                break;
            case "villain":
            case "hero":
                if (entity.shape === "circle") {
                    game.context.drawImage(entity.sprite, 0, 0, entity.sprite.width, entity.sprite.height,
                            -entity.radius - padding, -entity.radius - padding, entity.radius * 2 + 2 * padding, entity.radius * 2 + 2 * padding);
                } else if (entity.shape === "rectangle") {
                    game.context.drawImage(entity.sprite, 0, 0, entity.sprite.width, entity.sprite.height,
                            -entity.width / 2 - padding, -entity.height / 2 - padding, entity.width + 2 * padding, entity.height + 2 * padding);
                }
                break;
            case "ground":
                    // Nic nie robimy... Obiekty takie jak ground i slingshot narysujemy osobno
                break;
        }

        game.context.rotate(-angle);
        game.context.translate(-position.x * box2d.scale + game.offsetLeft, -position.y * box2d.scale);
    }

};

var box2d = {
    scale: 30,

    init: function() {
        // Definiujemy świat Box2D, który będzie wykonywać większość obliczeń fizycznych
        var gravity = new b2Vec2(0, 9.8); // Deklarujemy grawitację o wartości 9.8 m/s^2, skierowaną w dół
        var allowSleep = true; // Umożliwiamy zasypianie obiektów w spoczynku oraz ich wykluczanie z obliczeń

        box2d.world = new b2World(gravity, allowSleep);

        // Aktywujemy rysowanie w trybie debugowania. Aby je wyłączyć, należy umieścić poniższy wiersz w komentarzu.
        // this.setupDebugDraw();

        this.handleCollisions();
    },

    handleCollisions: function() {
        var listener = new b2ContactListener();

        listener.PostSolve = function(contact, impulse) {
            var body1 = contact.GetFixtureA().GetBody();
            var body2 = contact.GetFixtureB().GetBody();
            var entity1 = body1.GetUserData();
            var entity2 = body2.GetUserData();

            var impulseAlongNormal = Math.abs(impulse.normalImpulses[0]);

            // Ta funkcja jest wywoływana nieco zbyt często. Odrzucamy bardzo słabe impulsy.
            // Po wypróbowaniu różnych wartości, wydaje się, że dobrą wartością progową będzie 5
            if (impulseAlongNormal > 5) {
                // Jeśli obiekty mają właściwość health, zmniejszamy jej wartość o wartość impulsu
                if (entity1.health) {
                    entity1.health -= impulseAlongNormal;
                }

                if (entity2.health) {
                    entity2.health -= impulseAlongNormal;
                }

                // Jeśli do encji przydzielony jest dźwięk odbijania, odtwarzamy go
                if (entity1.bounceSound) {
                    entity1.bounceSound.play();
                }

                if (entity2.bounceSound) {
                    entity2.bounceSound.play();
                }
            }
        };

        box2d.world.SetContactListener(listener);
    },

    debugCanvas: undefined,
    setupDebugDraw: function() {
        // Dynamicznie tworzymy obiekt canvas, służący do rysowania w trybie debugowania
        if (!box2d.debugCanvas) {
            var canvas = document.createElement("canvas");

            canvas.width = 1024;
            canvas.height = 480;
            document.body.appendChild(canvas);
            canvas.style.top = "480px";
            canvas.style.position = "absolute";
            canvas.style.background = "white";
            box2d.debugCanvas = canvas;
        }

        // Konfigurujemy rysowanie w trybie debugowania
        var debugContext = box2d.debugCanvas.getContext("2d");
        var debugDraw = new b2DebugDraw();

        debugDraw.SetSprite(debugContext);
        debugDraw.SetDrawScale(box2d.scale);
        debugDraw.SetFillAlpha(0.3);
        debugDraw.SetLineThickness(1.0);
        debugDraw.SetFlags(b2DebugDraw.e_shapeBit | b2DebugDraw.e_jointBit);
        box2d.world.SetDebugDraw(debugDraw);
    },

    createRectangle: function(entity, definition) {
        var bodyDef = new b2BodyDef();

        if (entity.isStatic) {
            bodyDef.type = b2Body.b2_staticBody;
        } else {
            bodyDef.type = b2Body.b2_dynamicBody;
        }

        bodyDef.position.x = entity.x / box2d.scale;
        bodyDef.position.y = entity.y / box2d.scale;
        if (entity.angle) {
            bodyDef.angle = Math.PI * entity.angle / 180;
        }

        var fixtureDef = new b2FixtureDef();

        fixtureDef.density = definition.density;
        fixtureDef.friction = definition.friction;
        fixtureDef.restitution = definition.restitution;

        fixtureDef.shape = new b2PolygonShape();
        fixtureDef.shape.SetAsBox(entity.width / 2 / box2d.scale, entity.height / 2 / box2d.scale);

        var body = box2d.world.CreateBody(bodyDef);

        body.SetUserData(entity);
        body.CreateFixture(fixtureDef);

        return body;
    },

    createCircle: function(entity, definition) {
        var bodyDef = new b2BodyDef();

        if (entity.isStatic) {
            bodyDef.type = b2Body.b2_staticBody;
        } else {
            bodyDef.type = b2Body.b2_dynamicBody;
        }

        bodyDef.position.x = entity.x / box2d.scale;
        bodyDef.position.y = entity.y / box2d.scale;

        if (entity.angle) {
            bodyDef.angle = Math.PI * entity.angle / 180;
        }
        var fixtureDef = new b2FixtureDef();

        fixtureDef.density = definition.density;
        fixtureDef.friction = definition.friction;
        fixtureDef.restitution = definition.restitution;

        fixtureDef.shape = new b2CircleShape(entity.radius / box2d.scale);

        var body = box2d.world.CreateBody(bodyDef);

        body.SetUserData(entity);
        body.CreateFixture(fixtureDef);

        return body;
    },

    step: function(timeStep) {
        // Zgodnie z dokumentacją Box2D, jeśli wartość timeStep jest większa niż 1 / 30,
        // Box2D może mieć problemy z wykrywaniem kolizji
        // Dlatego przycinamy timeStep do 1 / 30

        if (timeStep > 2 / 60) {
            timeStep = 2 / 60;
        }

        // velocity iterations = 8
        // position iterations = 3

        box2d.world.Step(timeStep, 8, 3);
    }

};

// Inicjalizujemy grę dopiero po całkowitym wczytaniu strony
window.addEventListener("load", function() {
    game.resize();
    game.init();
});

window.addEventListener("resize", function() {
    game.resize();
});

document.addEventListener("touchmove", function(ev) {
    ev.preventDefault();
});
