-- Listing 3.17

CREATE PROCEDURE dbo.RebuildIndexes
(    @ShowOrRebuilId nvarchar(10) = N'pokaz'
   , @MaxFrag decimal(20, 2) = 20.0
)
AS
SET NOCOUNT ON;

BEGIN
-- Deklaracje zmiennych
DECLARE
   @Schema nvarchar(128), @Table nvarchar(128)
   , @Index nvarchar(128), @Sql nvarchar(4000)
   , @DatabaseId int, @SchemaId int
   , @TableId int, @lndexId int;
-- Tworzenie tabeli z list indeksw 
DECLARE @IndexList TABLE
(    DatabaseName nvarchar(128) NOT NULL
   , DatabaseId int NOT NULL
   , SchemaName nvarchar(128) NOT NULL
   , SchemaId int NOT NULL
   , TableName nvarchar(128) NOT NULL
   , TableId int NOT NULL
   , IndexName nvarchar(128)
   , IndexId int NOT NULL
   , Fragmentation decimal(20, 2)
   , PRIMARY KEY (DatabaseId, SchemaId, TableId, IndexId)
);

-- Wypenienie tabeli list indeksw 
INSERT INTO @IndexList
(    DatabaseName, DatabaseId
   , SchemaName, SchemaId
   , TableName, TableId
   , IndexName, IndexId
   , Fragmentation
)
 SELECT db_name(), db_id()
   , s.Name, s.schema_id
   , t.Name, t.object_id
   , i.Name, i.index_id
   , MAX(ip.avg_fragmentation_in_percent)
 FROM sys.tables t
INNER JOIN sys.schemas s ON
    t.schema_id = s.schema_id
INNER JOIN sys.indexes i ON
    t.object_id = i.object_id
INNER JOIN sys.dm_db_index_physical_stats (db_id(), NULL, NULL, NULL, NULL) ip ON
    ip.object_id = t.object_id AND ip.index_id = i.index_id
WHERE ip.database_id = db_id()
GROUP BY
    s.Name
    , s.schema_id
    , t.Name
    , t.object_id
    , i.Name
    , i.index_id;

-- Jeli wpisano polecenie 'odbuduj', korzystajc z kursora,
-- przejd przez wszystkie indeksy, i odbudowujc je 
IF @ShowOrRebuilId = N'odbuduj'
BEGIN

-- Zadeklaruj kursor do tworzenia dynamicznego wyraenia SQL 
DECLARE Index_Cursor CURSOR FAST_FORWARD
    FOR SELECT SchemaName, TableName, IndexName
        FROM @IndexList
        WHERE Fragmentation > @MaxFrag
        ORDER BY Fragmentation DESC, TableName ASC, IndexName ASC;

-- Otwrz kursor do odczytu 
OPEN Index_Cursor;
-- Przejd przez wszystkie tabele w bazie 
FETCH NEXT FROM Index_Cursor
        INTO @Schema, @Table, @Index;
WHILE @@FETCH_STATUS = 0
BEGIN -- Utwrz wyraenie ALTER INDEX odbudowujce indeks
    SET @Sql = N'ALTER INDEX ' +
        QUOTENAME(RTRIM(@Index)) + N' ON ' + QUOTENAME(RTRIM(@Table)) + N'.' +
        QUOTENAME(RTRIM(@Table)) + N' REBUILD WITH (ONLINE = OFF); ';

    PRINT @Sql;

    -- Wykonaj dynamiczny SQL 
    EXEC (@Sql);

    -- Pobierz kolejny indeks
    FETCH NEXT FROM Index_Cursor
    INTO @Schema, @Table, @Index;
END

-- Zamknij i zwolnij kursor
CLOSE Index_Cursor;
DEALLOCATE Index_Cursor;
END

-- Wywietl wyniki zawierajce informacje o fragmentacji przed odbudowaniem i po 
SELECT il.DatabaseName
     , il.SchemaName
     , il.TableName
     , il.IndexName
     , il.Fragmentation AS FragmentationStart
     , MAX( CAST(ip.avg_fragmentation_in_percent AS DECIMAL(20, 2))
         ) AS FragmentationEnd
FROM @IndexList il
INNER JOIN sys.dm_db_index_physical_stats(@DatabaseId, NULL, NULL, NULL, NULL) ip ON
    DatabaseId = ip.database_id AND
    TableId = ip.object_id AND
    IndexId = ip.index_id
GROUP BY
    il.DatabaseName
    , il.SchemaName
    , il.TableName
    , il.IndexName
    , il.Fragmentation
ORDER BY
    Fragmentation DESC
    , TableName ASC
    , IndexName ASC;
RETURN;
END
GO

-- Wykonaj procedur skadowan odbudowujc indeks 
EXEC dbo.RebuildIndexes N'odbuduj', 30;
