
//: C11:CarBuilder.cpp {RunByHand}
// Dziaanie wywoania broadcast()
//{L} ZThread
#include <iostream>
#include <string>
#include "zthread/Thread.h"
#include "zthread/Mutex.h"
#include "zthread/Guard.h"
#include "zthread/Condition.h"
#include "zthread/ThreadedExecutor.h"
#include "TQueue.h"
using namespace ZThread;
using namespace std;

class Car {
  int id;
  bool engine, driveTrain, wheels;
public:
  Car(int idn) : id(idn), engine(false),
  driveTrain(false), wheels(false) {}
  // :Puste obiekty Car:
  Car() : id(-1), engine(false),
  driveTrain(false), wheels(false) {}
  // Brak synchronizacji  zaoenie niepodzielnoci operacji na typie bool:
  int getId() { return id; }
  void addEngine() { engine = true; }
  bool engineInstalled() { return engine; }
  void addDriveTrain() { driveTrain = true; }
  bool driveTrainInstalled() { return driveTrain; }
  void addWheels() { wheels = true; }
  bool wheelsInstalled() { return wheels; }
  friend ostream& operator<<(ostream& os, const Car& c) {
    return os << "Samochd " << c.id << " ["
      << " silnik: " << c.engine
      << " napd: " << c.driveTrain
      << " koa: " << c.wheels << " ]";
  }
};

typedef CountedPtr< TQueue<Car> > CarQueue;

class ChassisBuilder : public Runnable {
  CarQueue carQueue;
  int counter;
public:
  ChassisBuilder(CarQueue& cq) : carQueue(cq),counter(0) {}
  void run() {
    try {
      while(!Thread::interrupted()) {
        Thread::sleep(1000);
        // Konstrukcja ramy:
        Car c(counter++);
        cout << c << endl;
        // Wstawienie ramy do kolejki
        carQueue->put(c);
      }
    } catch(Interrupted_Exception&) { /* Wyjcie */ }
    cout << "Modu spawania ramy wyczony" << endl;
  }
};

class Cradle {
  Car c; // Przechowuje aktualnie przetwarzany samochd
  bool occupied;
  Mutex workLock, readyLock;
  Condition workCondition, readyCondition;
  bool engineBotHired, wheelBotHired, driveTrainBotHired;
public:
  Cradle()
  : workCondition(workLock), readyCondition(readyLock) {
    occupied = false;
    engineBotHired = true;
    wheelBotHired = true;
    driveTrainBotHired = true;
  }
  void insertCar(Car chassis) {
    c = chassis;
    occupied = true;
  }
  Car getCar() { // Jednokrotne pobieranie samochodu z kolejki
    if(!occupied) {
      cerr << "Brak samochodu na stanowisku roboczym w getCar()" << endl;
      return Car(); // "Pusty" obiekt Car
    }
    occupied = false;
    return c;
  }
  // Odwoanie do samochodu na stanowisku robocznym:
  Car* operator->() { return &c; }
  // Rejestrowanie usug robotw dziaajcych przy stanowisku:
  void offerEngineBotServices() {
    Guard<Mutex> g(workLock);
    while(engineBotHired)
      workCondition.wait();
    engineBotHired = true; // Akceptacja robota
  }
  void offerWheelBotServices() {
    Guard<Mutex> g(workLock);
    while(wheelBotHired)
      workCondition.wait();
    wheelBotHired = true; // Akceptacja robota
  }
  void offerDriveTrainBotServices() {
    Guard<Mutex> g(workLock);
    while(driveTrainBotHired)
      workCondition.wait();
    driveTrainBotHired = true; // Akceptacja robota
  }
  // Poinformowanie oczekujcych robotw o zakoczeniu pracy:
  void startWork() {
    Guard<Mutex> g(workLock);
    engineBotHired = false;
    wheelBotHired = false;
    driveTrainBotHired = false;
    workCondition.broadcast();
  }
  // Kady robot zgasza koniec pracy:
  void taskFinished() {
    Guard<Mutex> g(readyLock);
    readyCondition.signal();
  }
  // Operator oczekuje na zakoczenie pracy wszystkich robotw:
  void waitUntilWorkFinished() {
    Guard<Mutex> g(readyLock);
    while(!(c.engineInstalled() && c.driveTrainInstalled()
            && c.wheelsInstalled()))
      readyCondition.wait();
  }
};

typedef CountedPtr<Cradle> CradlePtr;

class Director : public Runnable {
  CarQueue chassisQueue, finishingQueue;
  CradlePtr cradle;
public:
  Director(CarQueue& cq, CarQueue& fq, CradlePtr cr)
  : chassisQueue(cq), finishingQueue(fq), cradle(cr) {}
  void run() {
    try {
      while(!Thread::interrupted()) {
        // Blokowanie do momentu dostpnoci ramy:
        cradle->insertCar(chassisQueue->get());
        // Powiadomienie robotw o moliwoci rozpoczcia pracy
        cradle->startWork();
        // Oczekiwanie na zakoczenie pracy
        cradle->waitUntilWorkFinished();
        // Umieszczenie samochodu w kolejce wyjciowej
        finishingQueue->put(cradle->getCar());
      }
    } catch(Interrupted_Exception&) { /* Wyjcie */ }
    cout << "Modu Operatora wyczony" << endl;
  }
};

class EngineRobot : public Runnable {
  CradlePtr cradle;
public:
  EngineRobot(CradlePtr cr) : cradle(cr) {}
  void run() {
    try {
      while(!Thread::interrupted()) {
        // Blokowanie do momentu przyjcie (akceptacji) zadania:
        cradle->offerEngineBotServices();
        cout << "Instalowanie silnika" << endl;
        (*cradle)->addEngine();
        cradle->taskFinished();
      }
    } catch(Interrupted_Exception&) { /* Wyjcie */ }
    cout << "Robot montaowy silnika wyczony" << endl;
  }
};

class DriveTrainRobot : public Runnable {
  CradlePtr cradle;
public:
  DriveTrainRobot(CradlePtr cr) : cradle(cr) {}
  void run() {
    try {
      while(!Thread::interrupted()) {
        // Blokowanie do momentu przyjcie (akceptacji) zadania:
        cradle->offerDriveTrainBotServices();
        cout << "Instalacja mechanizmu przeniesienia napdu" << endl;
        (*cradle)->addDriveTrain();
        cradle->taskFinished();
      }
    } catch(Interrupted_Exception&) { /* Wyjcie */ }
    cout << "Robot montaowy napdu wyczony" << endl;
  }
};

class WheelRobot : public Runnable {
  CradlePtr cradle;
public:
  WheelRobot(CradlePtr cr) : cradle(cr) {}
  void run() {
    try {
      while(!Thread::interrupted()) {
        // Blokowanie do momentu przyjcie (akceptacji) zadania:
        cradle->offerWheelBotServices();
        cout << "Instalowanie k" << endl;
        (*cradle)->addWheels();
        cradle->taskFinished();
      }
    } catch(Interrupted_Exception&) { /* Wyjcie */ }
    cout << "Robot montaowy k wyczony" << endl;
  }
};

class Reporter : public Runnable {
  CarQueue carQueue;
public:
  Reporter(CarQueue& cq) : carQueue(cq) {}
  void run() {
    try {
      while(!Thread::interrupted()) {
        cout << carQueue->get() << endl;
      }
    } catch(Interrupted_Exception&) { /* Exit */ }
    cout << "Modu kontrolera wyczony" << endl;
  }
};

int main() {
  cout << "Aby zakoczy, nacinij klawisz <Enter>" << endl;
  try {
    CarQueue chassisQueue(new TQueue<Car>),
             finishingQueue(new TQueue<Car>);
    CradlePtr cradle(new Cradle);
    ThreadedExecutor assemblyLine;
    assemblyLine.execute(new EngineRobot(cradle));
    assemblyLine.execute(new DriveTrainRobot(cradle));
    assemblyLine.execute(new WheelRobot(cradle));
    assemblyLine.execute(
      new Director(chassisQueue, finishingQueue, cradle));
    assemblyLine.execute(new Reporter(finishingQueue));
    // Rozpoczcie caoci procesu od skonstruowania ramy:
    assemblyLine.execute(new ChassisBuilder(chassisQueue));
    cin.get();
    assemblyLine.interrupt();
  } catch(Synchronization_Exception& e) {
    cerr << e.what() << endl;
  }
} ///:~
