/// <reference types="cypress" />
// Wpisanie Cypress.sinon nie usuwa żadnych kontroli
// https://github.com/cypress-io/cypress/issues/6720

context('Spies, Stubs, and Clock', () => {
  it('cy.spy() - wrap a method in a spy', () => {
    // https://on.cypress.io/spy
    cy.visit('https://example.cypress.io/commands/spies-stubs-clocks')

    const obj = {
      foo () {},
    }

    const spy = cy.spy(obj, 'foo').as('anyArgs')

    obj.foo()

    expect(spy).to.be.called
  })

  it('cy.spy() retries until assertions pass', () => {
    cy.visit('https://example.cypress.io/commands/spies-stubs-clocks')

    const obj = {
      /**
       * Wypisuje przekazany argument
       * @param x {any}
      */
      foo (x) {
        console.log('obj.foo called with', x)
      },
    }

    cy.spy(obj, 'foo').as('foo')

    setTimeout(() => {
      obj.foo('first')
    }, 500)

    setTimeout(() => {
      obj.foo('second')
    }, 2500)

    cy.get('@foo').should('have.been.calledTwice')
  })

  it('cy.stub() - create a stub and/or replace a function with stub', () => {
    // https://on.cypress.io/stub
    cy.visit('https://example.cypress.io/commands/spies-stubs-clocks')

    const obj = {
      /**
       * Wypisuje oba argumenty
       * @param a {string}
       * @param b {string}
      */
      foo (a, b) {
        console.log('a', a, 'b', b)
      },
    }

    const stub = cy.stub(obj, 'foo').as('foo')

    obj.foo('foo', 'bar')

    expect(stub).to.be.called
  })

  it('cy.clock() - control time in the browser', () => {
    // https://on.cypress.io/clock

    // Tworzymy datą w UTC, aby zawsze była taka sama,
    // bez względu na to, w jakiej lokalnej strefie czasowej działa przeglądarka
    const now = new Date(Date.UTC(2017, 2, 14)).getTime()

    cy.clock(now)
    cy.visit('https://example.cypress.io/commands/spies-stubs-clocks')
    cy.get('#clock-div').click()
      .should('have.text', '1489449600')
  })

  it('cy.tick() - move time in the browser', () => {
    // https://on.cypress.io/tick

    // Tworzymy datą w UTC, aby zawsze była taka sama,
    // bez względu na to, w jakiej lokalnej strefie czasowej działa przeglądarka
    const now = new Date(Date.UTC(2017, 2, 14)).getTime()

    cy.clock(now)
    cy.visit('https://example.cypress.io/commands/spies-stubs-clocks')
    cy.get('#tick-div').click()
      .should('have.text', '1489449600')

    cy.tick(10000) // Upłynęło 10 sekund
    cy.get('#tick-div').click()
      .should('have.text', '1489449610')
  })

  it('cy.stub() matches depending on arguments', () => {
    // Wszystkie dostępne matchery znajdziesz na stronie
    // https://sinonjs.org/releases/latest/matchers/
    const greeter = {
      /**
       * Pozdrawianie
       * @param {string} name
      */
      greet (name) {
        return `Hello, ${name}!`
      },
    }

    cy.stub(greeter, 'greet')
      .callThrough() // Jeśli chcesz, by wywołania bez dopasowania wywoływały prawdziwą metodę 
      .withArgs(Cypress.sinon.match.string).returns('Hi')
      .withArgs(Cypress.sinon.match.number).throws(new Error('Invalid name'))

    expect(greeter.greet('World')).to.equal('Hi')
    // @ts-ignore
    expect(() => greeter.greet(42)).to.throw('Invalid name')
    expect(greeter.greet).to.have.been.calledTwice

    // Wywołania bez dopasowania są kierowane do prawdziwej metody
    // @ts-ignore
    expect(greeter.greet()).to.equal('Hello, undefined!')
  })

  it('matches call arguments using Sinon matchers', () => {
    // Wszystkie dostępne matchery znajdziesz na stronie
    // https://sinonjs.org/releases/latest/matchers/
    const calculator = {
      /**
       * Zwraca sumę dwóch argumentów
       * @param a {number}
       * @param b {number}
      */
      add (a, b) {
        return a + b
      },
    }

    const spy = cy.spy(calculator, 'add').as('add')

    expect(calculator.add(2, 3)).to.equal(5)

    // Jeśli chcemy przeprowadzić asercję wartości użytych podczas wywołania
    expect(spy).to.be.calledWith(2, 3)

    // Sprawdzamy, czy metoda "add" została wywołana z dwiema liczbami
    expect(spy).to.be.calledWith(Cypress.sinon.match.number, Cypress.sinon.match.number)

    // Ewentualnie podajemy wartość do dopasowania
    expect(spy).to.be.calledWith(Cypress.sinon.match(2), Cypress.sinon.match(3))

    // Dopasowujemy każdą wartość
    expect(spy).to.be.calledWith(Cypress.sinon.match.any, 3)

    // Dopasowujemy każdą wartość z listy
    expect(spy).to.be.calledWith(Cypress.sinon.match.in([1, 2, 3]), 3)

    /**
     * Zwraca true, jeżeli podana liczba jest zdarzeniem
     * @param {number} x
     */
    const isEven = (x) => x % 2 === 0

    // Oczekujemy, że wartość zostanie przekazana do standardowej funkcji predykatu
    // Drigi argument funkcji "sinon.match(predicate, message)" jest wyświetlany,
    // jeśli predykat nie zostanie zakończony powodzeniem i asercja się nie powiedzie
    expect(spy).to.be.calledWith(Cypress.sinon.match(isEven, 'isEven'), 3)

    /**
     * Zraca funkcję, która sprawdza, czy dana liczba jest większa niż określony limit
     * @param {number} limit
     * @returns {(x: number) => boolean}
     */
    const isGreaterThan = (limit) => (x) => x > limit

    /**
     * Zraca funkcję, która sprawdza, czy dana liczba jest mniejsza niż określony limit
     * @param {number} limit
     * @returns {(x: number) => boolean}
     */
    const isLessThan = (limit) => (x) => x < limit

    // Używając "and" lub "or", możesz połączyć kilka matcherów
    expect(spy).to.be.calledWith(
      Cypress.sinon.match.number,
      Cypress.sinon.match(isGreaterThan(2), '> 2').and(Cypress.sinon.match(isLessThan(4), '< 4')),
    )

    expect(spy).to.be.calledWith(
      Cypress.sinon.match.number,
      Cypress.sinon.match(isGreaterThan(200), '> 200').or(Cypress.sinon.match(3)),
    )

    // Matchery można stosować w asercjach BDD
    cy.get('@add').should('have.been.calledWith',
      Cypress.sinon.match.number, Cypress.sinon.match(3))

    // Aby skrócić kod testowy, możesz tworzyć aliasy matcherów
    const { match: M } = Cypress.sinon

    cy.get('@add').should('have.been.calledWith', M.number, M(3))
  })
})
