from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

import numpy as np
import tensorflow as tf

tf.logging.set_verbosity(tf.logging.INFO)


def cnn_model_fn(features, labels, mode):
  """Funkcja modeli dla CNN."""
  # Warstwa wejściowa
  # Zmień kształt X na tensor 4-D: [batch_size, width, height, channels]
  # Obrazy MNIST maja rozmiar 28x28 pikseli i mają jeden kanał koloru
  input_layer = tf.reshape(features["x"], [-1, 28, 28, 1])

  # Warstwa splotowa nr 1
  # Oblicza 32 cechy przy użyciu filtra 5x5 a aktywacją ReLU.
  # Dodano uzupełnianie, aby zachować szerokość i wysokość.
  # Kształt tensora wejściowego: [batch_size, 28, 28, 1]
  # Kształt tensora wyjściowego: [batch_size, 28, 28, 32]
  conv1 = tf.layers.conv2d(
      inputs=input_layer,
      filters=32,
      kernel_size=[5, 5],
      padding="same",
      activation=tf.nn.relu)

  # Warstwa łącząca nr 1
  # Pierwsza maksymalizująca warstwa łącząca z filtrem 2x2 i krokiem 2
  # Kształt tensora wejściowego: [batch_size, 28, 28, 32]
  # Kształt tensora wyjściowego: [batch_size, 14, 14, 32]
  pool1 = tf.layers.max_pooling2d(inputs=conv1, pool_size=[2, 2], strides=2)

  # Warstwa splotowa nr 2
  # Oblicza 64 cechy przy użyciu filtra 5x5.
  # Dodano uzupełnianie, aby zachować szerokość i wysokość.
  # Kształt tensora wejściowego: [batch_size, 14, 14, 32]
  # Kształt tensora wyjściowego: [batch_size, 14, 14, 64]
  conv2 = tf.layers.conv2d(
      inputs=pool1,
      filters=64,
      kernel_size=[5, 5],
      padding="same",
      activation=tf.nn.relu)

  # Warstwa łącząca nr 2
  # Druga maksymalizująca warstwa łącząca z filtrem 2x2 i krokiem 2
  # Kształt tensora wejściowego: [batch_size, 14, 14, 64]
  # Kształt tensora wyjściowego: [batch_size, 7, 7, 64]
  pool2 = tf.layers.max_pooling2d(inputs=conv2, pool_size=[2, 2], strides=2)

  # Tensor spłaszczony w grupę wektorów
  # Kształt tensora wejściowego: [batch_size, 7, 7, 64]
  # Kształt tensora wyjściowego: [batch_size, 7 * 7 * 64]
  pool2_flat = tf.reshape(pool2, [-1, 7 * 7 * 64])

  # Gęsta warstwa
  # Gęsta warstwa łącząca z 1024 neuronami
  # Kształt tensora wejściowego: [batch_size, 7 * 7 * 64]
  # Kształt tensora wyjściowego: [batch_size, 1024]
  dense = tf.layers.dense(inputs=pool2_flat, units=1024, activation=tf.nn.relu)

  # Dodanie operacji porzucania; prawdopodobieństwo 0,6, że element będzie zatrzymany
  dropout = tf.layers.dropout(
      inputs=dense, rate=0.4, training=mode == tf.estimator.ModeKeys.TRAIN)

  # Warstwa logitowa
  # Kształt tensora wejściowego: [batch_size, 1024]
  # Kształt tensora wyjściowego: [batch_size, 10]
  logits = tf.layers.dense(inputs=dropout, units=10)

  predictions = {
      # Generowanie predykcji (dla trybu PREDICT i EVAL)
      "classes": tf.argmax(input=logits, axis=1),
      # Dodanie do grafu `softmax_tensor`. Używane dla PREDICT i przez 
      # `logging_hook`.
      "probabilities": tf.nn.softmax(logits, name="softmax_tensor")
  }
  if mode == tf.estimator.ModeKeys.PREDICT:
    return tf.estimator.EstimatorSpec(mode=mode, predictions=predictions)

  # Obliczanie straty (zarówno dla trybu TRAIN, jak i EVAL)
  loss = tf.losses.sparse_softmax_cross_entropy(labels=labels, logits=logits)

  # Konfigurowanie operacji szkolenia (dla trybu TRAIN)
  if mode == tf.estimator.ModeKeys.TRAIN:
    optimizer = tf.train.GradientDescentOptimizer(learning_rate=0.001)
    train_op = optimizer.minimize(
        loss=loss,
        global_step=tf.train.get_global_step())
    return tf.estimator.EstimatorSpec(mode=mode, loss=loss, train_op=train_op)

  # Dodanie wskaźnika ewaluacji (dla trybu EVAL)
  eval_metric_ops = {
      "accuracy": tf.metrics.accuracy(
          labels=labels, predictions=predictions["classes"])}
  return tf.estimator.EstimatorSpec(
      mode=mode, loss=loss, eval_metric_ops=eval_metric_ops)


def main(args, ctx):
  # Wczytanie danych treningowych i ewaluacyjnych
  mnist = tf.contrib.learn.datasets.mnist.read_data_sets(args.data_dir)
  train_data = mnist.train.images  # Zwraca np.array
  train_labels = np.asarray(mnist.train.labels, dtype=np.int32)
  eval_data = mnist.test.images  # Zwraca np.array
  eval_labels = np.asarray(mnist.test.labels, dtype=np.int32)

  # Utworzenie estymatora
  mnist_classifier = tf.estimator.Estimator(
      model_fn=cnn_model_fn, model_dir=args.model)

  # Ustawienie rejestrowania dla predykcji
  # Rejestrowanie wartości w tensorze "Softmax" z etykietą "probabilities"
  tensors_to_log = {"probabilities": "softmax_tensor"}
  logging_hook = tf.train.LoggingTensorHook(
      tensors=tensors_to_log, every_n_iter=50)

  # Szkolenie modelu
  train_input_fn = tf.estimator.inputs.numpy_input_fn(
      x={"x": train_data},
      y=train_labels,
      batch_size=args.batch_size,
      num_epochs=None,
      shuffle=True)
  

  # Ewaluacja modelu i wypisanie wyników
  eval_input_fn = tf.estimator.inputs.numpy_input_fn(
      x={"x": eval_data},
      y=eval_labels,
      num_epochs=1,
      shuffle=False)
  # eval_results = mnist_classifier.evaluate(input_fn=eval_input_fn)
  # print(eval_results)

  # Wykorzystanie tf.estimator.train_and_evaluate
  train_spec = tf.estimator.TrainSpec(input_fn=train_input_fn, max_steps=args.steps, hooks=[logging_hook])
  eval_spec = tf.estimator.EvalSpec(input_fn=eval_input_fn)
  tf.estimator.train_and_evaluate(mnist_classifier, train_spec, eval_spec)


if __name__ == "__main__":
  # tf.app.run()

  from pyspark.context import SparkContext
  from pyspark.conf import SparkConf
  from tensorflowonspark import TFCluster
  import argparse

  sc = SparkContext(conf=SparkConf().setAppName("mnist_spark"))
  executors = sc._conf.get("spark.executor.instances")
  num_executors = int(executors) if executors is not None else 1

  parser = argparse.ArgumentParser()
  parser.add_argument("--batch_size", help="liczba rekordów na grupę", type=int, default=100)
  parser.add_argument("--cluster_size", help="liczba węzłów w klastrze", type=int, default=num_executors)
  parser.add_argument("--data_dir", help="ścieżka dostępu do danych MNIST", default="MNIST-data")
  parser.add_argument("--model", help="ścieżka do zapisu modelu/punktów kontrolnych", default="mnist_model")
  parser.add_argument("--num_ps", help="number of PS nodes in cluster", type=int, default=1)
  parser.add_argument("--steps", help="maximum number of steps", type=int, default=1000)
  parser.add_argument("--tensorboard", help="launch tensorboard process", action="store_true")

  args = parser.parse_args()
  print("argumenty:", args)

  cluster = TFCluster.run(sc, main, args, args.cluster_size, args.num_ps, tensorboard=args.tensorboard, input_mode=TFCluster.InputMode.TENSORFLOW, log_dir=args.model, master_node='master')
  cluster.shutdown()