﻿using SpeakerMeet.Exceptions;
using SpeakerMeet.Models;
using SpeakerMeet.Repositories.Interfaces;
using Xunit;

namespace SpeakerMeet.Repositories.Tests.SpeakerRepositoryTests
{
    [Trait("Category", "SpeakerRepository")]
    public class Update
    {
        [Fact]
        public void ItHasUpdateMethod()
        {
            // Arrange
            IRepository<Speaker> repo = new SpeakerRepository();
            var speaker = repo.Create(new Speaker());

            // Act
            var result = repo.Update(speaker);
        }

        [Fact]
        public void ItUpdatesASpeaker()
        {
            // Arrange
            var repo = new SpeakerRepository();
            var speaker = repo.Create(new Speaker { Name = "Testowa nazwa" });
            speaker.Name = "Nowa Nazwa";

            // Act
            var result = repo.Update(speaker);

            // Assert
            Assert.Equal(speaker.Name, result.Name);
        }

        [Fact]
        public void ItUpdatesASpeakerInTheRepository()
        {
            // Arrange
            var repo = new SpeakerRepository();
            var speaker = repo.Create(new Speaker { Name = "Testowa nazwa" });
            speaker.Name = "Nowa nazwa";

            // Act
            var updatedSpeaker = repo.Update(speaker);

            // Audit
            var result = repo.Get(speaker.Id);

            // Assert
            Assert.Equal("Nowa nazwa", result.Name);
        }

        [Fact]
        public void ItThrowsNotFoundExceptionWhenSpeakerDoesNotExist()
        {
            // Arrange
            var repo = new SpeakerRepository();
            var speaker = new Speaker { Id = 5, Name = "Testowa nazwa" };

            // Act
            var result = Record.Exception(() => repo.Update(speaker));

            // Assert
            Assert.IsAssignableFrom<SpeakerNotFoundException>(result.GetBaseException());
        }

        [Fact]
        public void ItProtectsAgainstObjectChanges()
        {
            // Arrange
            var repo = new SpeakerRepository();
            var speaker = repo.Create(new Speaker { Name = "Testowa nazwa" });
            speaker.Name = "Nowa nazwa";
            var updatedSpeaker = repo.Update(speaker);

            // Act
            updatedSpeaker.Name = "Zaktualizowana Nazwa";

            // Audit
            var result = repo.Get(updatedSpeaker.Id);

            // Assert
            Assert.NotEqual("Zaktualizowana Nazwa", result.Name);
        }

        [Fact]
        public void ItProtectsAgainstOriginalObjectChanges()
        {
            // Arrange
            var repo = new SpeakerRepository();
            var speaker = repo.Create(new Speaker { Name = "Testowa Nazwa" });
            speaker.Name = "Nowa nazwa";
            var updatedSpeaker = repo.Update(speaker);

            // Act
            speaker.Name = "Zaktualizowana Nazwa";

            // Audit
            var result = repo.Get(updatedSpeaker.Id);

            // Assert
            Assert.NotEqual("Zaktualizowana Nazwa", result.Name);
        }
    }
}
