using System.Collections.Generic;
using System.Linq;
using Microsoft.AspNetCore.Mvc;
using Moq;
using SpeakerMeet.API.Controllers;
using SpeakerMeet.DTO;
using SpeakerMeet.Services.Interfaces;
using Xunit;

namespace SpeakerMeet.API.Tests.SpeakerControllerTests
{
    public class Search
    {
        private readonly SpeakerController _controller;
        private static Mock<ISpeakerService> _speakerServiceMock;
        private readonly List<Speaker> _speakers;

        public Search()
        {
            _speakers = new List<Speaker> { new Speaker
            {
                Name = "test"
            } };

            // define the mock
            _speakerServiceMock = new Mock<ISpeakerService>();

            // when search is called, return list of speakers containing speaker
            _speakerServiceMock.Setup(x => x.Search(It.IsAny<string>()))
                .Returns(() => _speakers);

            // pass mock object as ISpeakerService
            _controller = new SpeakerController(_speakerServiceMock.Object);
        }

        [Fact]
        public void ItReturnsCollectionOfSpeakers()
        {
            // Arrange
            // Act
            var result = _controller.Search("Jan") as OkObjectResult;

            // Assert
            Assert.NotNull(result);
            Assert.NotNull(result.Value);
            Assert.IsType<List<Speaker>>(result.Value);
        }

        [Fact(Skip = "Ju niepotrzebny")]
        public void GivenExactMatchThenOneSpeakerInCollection()
        {
            // Arrange
            // Act
            var result = _controller.Search("Janusz") as OkObjectResult;

            // Assert  
            var speakers = ((IEnumerable<Speaker>)result.Value).ToList();
            Assert.Equal(1, speakers.Count);
            Assert.Equal("Janusz", speakers[0].Name);
        }

        [Theory(Skip = "Ju niepotrzebny")]
        [InlineData("Janusz")]
        [InlineData("janusz")]
        [InlineData("JaNuSz")]
        public void GivenCaseInsensitveMatchThenSpeakerInCollection(string searchString)
        {
            // Arrange
            // Act
            var result = _controller.Search(searchString) as OkObjectResult;

            // Assert
            var speakers = ((IEnumerable<Speaker>)result.Value).ToList();
            Assert.Equal(1, speakers.Count);
            Assert.Equal("Janusz", speakers[0].Name);
        }

        [Fact(Skip = "Ju niepotrzebny")]
        public void GivenNoMatchThenEmptyCollection()
        {
            // Arrange
            // Act
            var result = _controller.Search("ZZZ") as OkObjectResult;

            // Assert
            var speakers = ((IEnumerable<Speaker>)result.Value).ToList();
            Assert.Equal(0, speakers.Count);
        }

        [Fact(Skip = "Ju niepotrzebny")]
        public void Given3MatchThenCollectionWith3Speakers()
        {
            // Arrange
            // Act 
            var result = _controller.Search("jan") as OkObjectResult;

            // Assert  
            var speakers = ((IEnumerable<Speaker>)result.Value).ToList();
            Assert.Equal(3, speakers.Count);
            Assert.True(speakers.Any(s => s.Name == "Jan"));
            Assert.True(speakers.Any(s => s.Name == "Janusz"));
            Assert.True(speakers.Any(s => s.Name == "Janina"));
        }

        [Fact(Skip = "Ju niepotrzebny")]
        public void ItAcceptsInterface()
        {
            // Arrange 
            ISpeakerService testSpeakerService = new TestSpeakerService();

            // Act
            var controller = new SpeakerController(testSpeakerService);

            // Assert
            Assert.NotNull(controller);
        }

        [Fact(Skip = "Ju niepotrzebny")]
        public void ItCallsSearchServiceOnce()
        {
            // Arrange
            // Act
            _controller.Search("jan");

            // Assert
            _speakerServiceMock.Verify(mock => mock.Search(It.IsAny<string>()),
                Times.Once());
        }

        [Fact]
        public void GivenSearchStringThenSpeakerServiceSearchCalledWithString()
        {
            // Arrange
            var searchString = "jan";

            // Act
            _controller.Search(searchString);

            // Assert
            _speakerServiceMock.Verify(mock => mock.Search(searchString),
                Times.Once());
        }

        [Fact]
        public void GivenSpeakerServiceThenResultsReturned()
        {
            // Arrange
            var searchString = "jan";

            // Act 
            var result = _controller.Search(searchString) as OkObjectResult;

            // Assert
            Assert.NotNull(result);
            var speakers = ((IEnumerable<Speaker>)result.Value).ToList();
            Assert.Equal(_speakers, speakers);
        }
    }
}
