import {
  expect
} from 'chai';
import * as speakerActions from './speakerActions';
import * as types from '../reducers/actionTypes';
import * as errorTypes from '../reducers/errorTypes';
import thunk from 'redux-thunk';
import configureMockStore from 'redux-mock-store';
import sinon from 'sinon';
import factory from '../services/serviceFactory';
import { MockSpeakerService } from '../services/mockSpeakerService';

describe('Akcje prelegenta', () => {
  describe('Akcje synchroniczne', () => {
    describe('Pomyślne pobranie prelegentów', () => {
      it('istnieje', () => {
        expect(speakerActions.getSpeakersSuccess).to.exist;
      });

      it('jest tworzony z poprawnymi danymi', () => {
        // arrange
        const speakers = [{
          id: 'testowy-prelegent',
          firstName: 'Testowy',
          lastName: 'Prelegent'
        }];

        // act
        const result = speakerActions.getSpeakersSuccess(speakers);

        // assert
        expect(result.type).to.equal(types.GET_SPEAKERS_SUCCESS);
        expect(result.speakers).to.have.lengthOf(1);
        expect(result.speakers).to.deep.equal(speakers);
      });
    });

    describe('Pomyślne odnalezienie prelegenta', () => {
      it('istnieje', () => {
        expect(speakerActions.getSpeakerSuccess).to.exist;
      });

      it('jest tworzony z poprawnymi danymi', () => {
        // arrange
        const speaker = {
          id: 'testowy-prelegent',
          firstName: 'Testowy',
          lastName: 'Prelegent'  
        };

        // act
        const result = speakerActions.getSpeakerSuccess(speaker);

        // assert
        expect(result.type).to.equal(types.GET_SPEAKER_SUCCESS);
        expect(result.speaker).to.deep.equal(speaker);
      });
    });

    describe('Nieudane pobranie prelegenta', () => {
      it('istnieje', () => {
        expect(speakerActions.getSpeakerFailure).to.exist;
      });

      it('jest tworzony z poprawnymi danymi', () => {
        // arrange
        const error = {
          type: errorTypes.SPEAKER_NOT_FOUND
        };

        // act
        const result = speakerActions.getSpeakerFailure(error);

        // assert
        expect(result.type).to.equal(types.GET_SPEAKER_FAILURE);
        expect(result.error).to.deep.equal(error);
      });
    });
  });

  describe('Akcje asynchroniczne', () => {
    const middleware = [thunk];
    let mockStore;
    let create;
    let getAll;
    let getById;

    beforeEach(() => {
      let service = factory.createSpeakerService();
      let mockService = new MockSpeakerService();

      getAll = sinon.stub(service, "getAll");
      getAll.callsFake(mockService.getAll.bind(mockService));

      getById = sinon.stub(service, "getById");
      getById.callsFake(mockService.getById.bind(mockService));

      create = sinon.stub(service, "create");
      create.callsFake(mockService.create.bind(mockService));

      mockStore = configureMockStore(middleware);
    });

    afterEach(() => {
      create.restore();
      getAll.restore();
      getById.restore();
    });

    describe('Pobieranie prelegentów', () => {
      it('istnieje', () => {
        expect(speakerActions.getSpeakers).to.exist;
      });

      it('tworzy GET_SPEAKERS_SUCCESS podczas ładowania prelegentów', () => {
        // arrange
        const speaker = {
          id: 'testowy-prelegent',
          firstName: 'Testowy',
          lastName: 'Prelegent'  
        };
        const expectedActions = [speakerActions.getSpeakersSuccess([speaker])];
        const store = mockStore({
          speakers: []
        });

        // act
        return store.dispatch(speakerActions.getSpeakers()).then(() => {
          const actions = store.getActions();

          // assert
          expect(actions[0].type).to.equal(types.GET_SPEAKERS_SUCCESS);
        });
      });
    });

    describe('Pobieranie prelegenta', () => {
      it('istnieje', () => {
        expect(speakerActions.getSpeaker).to.exist;
      });

      it('tworzy GET_SPEAKER_FAILURE, kiedy prelegent nie zostanie odnaleziony', () => {
        // arrange
        const speakerId= 'nie-znaleziony-prelegent';
        const store = mockStore({
          speaker: {}
        });

        // act
        return store.dispatch(speakerActions.getSpeaker(speakerId)).then(() => {
          const actions = store.getActions();

          // assert
          expect(actions[0].type).to.equal(types.GET_SPEAKER_FAILURE);
        });
      });

      it('tworzy GET_SPEAKER_SUCCESS, kiedy prelegent zostanie znaleziony', () => {
        // arrange
        const speaker = {
          id: 'testowy-prelegent',
          firstName: 'Testowy',
          lastName: 'Prelegent'  
        };
        const store = mockStore({
          speaker: {}
        });
        let service = factory.createSpeakerService();
        service.create(speaker);

        // act
        return store.dispatch(speakerActions.getSpeaker('testowy-prelegent')).then(() => {
          const actions = store.getActions();

          // assert
          expect(actions[0].type).to.equal(types.GET_SPEAKER_SUCCESS);
          expect(actions[0].speaker.id).to.equal('testowy-prelegent');
          expect(actions[0].speaker.firstName).to.equal('Testowy');
          expect(actions[0].speaker.lastName).to.equal('Prelegent');
        });
      });
    });
  });
});
