import datetime
import os
from jose import jwt
from model.user import User
if os.getenv("CRYPTID_UNIT_TEST"):
    from fake import user as data
else:
    from data import user as data

from passlib.context import CryptContext

TOKEN_EXPIRES = 15 # minutes

# --- New auth stuff

# Change SECRET_KEY for production!
SECRET_KEY = "keep-it-secret-keep-it-safe"
ALGORITHM = "HS256"
pwd_context = CryptContext(schemes=["bcrypt"], deprecated="auto")

def verify_password(plain: str, hash: str) -> bool:
    """Utwórz skrót <plain> i powrócnaj z <hash> z bazy danych"""
    return pwd_context.verify(plain, hash)

def get_hash(plain: str) -> str:
    """Zwróć skrót łańcucha <plain>"""
    return pwd_context.hash(plain)

def get_jwt_username(token:str) -> str | None:
    """Zwróć nazwę użytkownika z znacznika dostępu JWT <token>"""
    try:
        payload = jwt.decode(token, SECRET_KEY, algorithms=[ALGORITHM])
        if not (username := payload.get("sub")):
            return None
    except jwt.JWTError:
        return None
    return username
    
def get_current_user(token: str) -> User | None:
    """Rozszyfruj znacznik dostępu OAuth <token> i zwróć obiekt User"""
    if not (username := get_jwt_username(token)):
        return None
    if (user := lookup_user(username)):
        return user
    return None
    
def lookup_user(name: str) -> User | None:
    """Zwróć obiekt User pasujący do <name> w bazie danych"""
    if (user := data.get(name)):
        return user
    return None

def auth_user(name: str, plain: str) -> User | None:
    """Uwierzytelnij użytkownika <name> i hasło <plain>"""
    if not (user := lookup_user(name)):
        return None
    if not verify_password(plain, user.hash):
        return None
    return user

def create_access_token(data: dict,
    expires: datetime.timedelta | None = None
):
    """Zwróć znacznik dostępu JWT"""
    src = data.copy()
    now = datetime.utcnow()
    expires = expires or datetime.timedelta(minutes=TOKEN_EXPIRES)
    src.update({"exp": now + expires})
    encoded_jwt = jwt.encode(src, SECRET_KEY, algorithm=ALGORITHM)
    return encoded_jwt
    
# --- CRUD passthrough stuff

def get_all() -> list[User]:
    return data.get_all()

def get_one(name) -> User:
    return data.get_one(name)

def create(user: User) -> User:
    return data.create(user)

def modify(name: str, user: User) -> User:
    return data.modify(name, user)

def delete(name: str) -> None:
    return data.delete(name)
