#include <optional>
#include <optional>
#include <queue>
#include <mutex>
#include <shared_mutex>
#include <unistd.h>
#include <string.h>

#include "spinlock.h"

#include "benchmark/benchmark.h"

template <typename T> void reset(T& q) { T().swap(q); }

struct LS {
    long x[1024];
    LS(char i = 0) { for (size_t k = 0; k < 1024; ++k) x[k] = i; }
};

// Kolejka cykliczna o stałej wielkości.
// Klasa nie jest bezpieczna wątkowo. Ma być przetwarzana w danej chwili
// przez jeden wątek.
template <typename T> class subqueue {
    public:
    explicit subqueue(size_t capacity) : capacity_(capacity), begin_(0), size_(0) {}
    size_t capacity() const { return capacity_; }
    size_t size() const { return size_; }
    bool push(const T& x) {
        if (size_ == capacity_) return false;
        size_t end = begin_ + size_++;
        if (end >= capacity_) end -= capacity_;
        data_[end] = x;
        return true;
    }
    bool pop(std::optional<T>& x) {
        if (size_ == 0) return false;
        --size_;
        size_t pos = begin_;
        if (++begin_ == capacity_) begin_ -= capacity_;
        x.emplace(std::move(data_[pos]));
        data_[pos].~T();
        return true;
    }
    static size_t memsize(size_t capacity) {
        return sizeof(subqueue) + capacity*sizeof(T);
    }
    static subqueue* construct(size_t capacity) {
        return new(::malloc(subqueue::memsize(capacity))) subqueue(capacity);
    }
    static void destroy(subqueue* queue) {
        queue->~subqueue();
        ::free(queue);
    }
    void reset() { 
        begin_ = size_ = 0;
    }

    private:
    const size_t capacity_;
    size_t begin_;
    size_t size_;
    T data_[1]; // Właściwie [capacity_]
};

// Kolekcja kilku podkolejek do optymalizacji dostępu współbieżnego.
// Każdy wskaźnik kolejki znajduje się w osobnym wierszu bufora.
template <typename Q> struct subqueue_ptr {
  subqueue_ptr() : queue() {}
  std::atomic<Q*> queue;
  char padding[64 - sizeof(queue)]; // Wypełnienie do wiersza bufora.
};

template <typename T> class concurrent_queue {
  typedef subqueue<T> subqueue_t;
  typedef subqueue_ptr<subqueue_t> subqueue_ptr_t;
  public:
    explicit concurrent_queue(size_t capacity = 1UL << 15) {
      for (int i = 0; i < QUEUE_COUNT; ++i) {
        queues_[i].queue.store(subqueue_t::construct(capacity), std::memory_order_relaxed);
      }
    }
    ~concurrent_queue() {
      for (int i = 0; i < QUEUE_COUNT; ++i) {
        subqueue_t* queue = queues_[i].queue.exchange(nullptr, std::memory_order_relaxed);
        subqueue_t::destroy(queue);
      }
    }

    // Ile pozycji jest w kolejce?
    size_t size() const { return count_.load(std::memory_order_acquire); }

    // Czy kolejka jest pusta?
    bool empty() const { return size() == 0; }

    // Uzyskanie elementu z kolejki.
    // Metoda ta powoduje blokowanie albo do chwili opróżnienia kolejki (size() == 0) albo
    // zwrócenia elementu z jednej z podkolejek.
    std::optional<T> pop() {
      std::optional<T> res;
      if (count_.load(std::memory_order_acquire) == 0) {
        return res;
      }
      // Przejęcie własności podkolejki. Dla wskaźnika podkolejki jest ponownie ustawiana
	  // wartość NULL, gdy wątek wywołujący jest w posiadaniu podkolejki. Po zrealizowaniu
	  // działania przywracanie wskaźnika powoduje wycofanie własności. Dostępna podkolejka
	  // może być pusta, ale nie oznacza to, że nie ma żadnych pozycji. Konieczne jest
      // sprawdzenie innych kolejek. Pętla może zostać zakończona po uzyskaniu elementu
	  // albo wtedy, gdy licznik elementów wskazuje na brak pozycji.
	  // Zauważ, że dekrementacja licznika nie jest atomowa w przypadku usuwania pozycji
	  // z kolejki, dlatego przez krótki czas blokada wirująca może być używana dla pustych
	  // kolejek do momentu wyrównania wartości licznika.
      
      subqueue_t* queue = NULL;
      bool success = false;
      for (size_t i = 0; ;)
      {
        i = dequeue_slot_.fetch_add(1, std::memory_order_relaxed) & (QUEUE_COUNT - 1);
        queue = queues_[i].queue.exchange(nullptr, std::memory_order_acquire);  // Przejmowanie własności podkolejki
        if (queue) {
          success = queue->pop(res);                                            // Usuwanie elementu z kolejki po przejęciu do niej 
		                                                                        // prawa własności
          queues_[i].queue.store(queue, std::memory_order_release);             // Wycofanie prawa własności
          if (success) break;                                                   // Uzyskanie elementu
          if (count_.load(std::memory_order_acquire) == 0) goto EMPTY;          // Brak elementu i brak kolejnych
        } else {                                                                // Kolejka ma wartość NULL. Nic nie musi być wycofywane
          if (count_.load(std::memory_order_acquire) == 0) goto EMPTY;          // Nie uzyskano dostępu do kolejki, ale nie ma już żadnych
                                                                                // pozycji
        }
        if (success) break;
        if (count_.load(std::memory_order_acquire) == 0) goto EMPTY;
        static const struct timespec ns = { 0, 1 };
        nanosleep(&ns, NULL);
      };
      // Jeśli istnieje element, zmniejszana jest wartość licznika elementów w kolejce.
      count_.fetch_add(-1);
EMPTY:
      return success;
    }

    bool push(const T& v) {
      count_.fetch_add(1);
      subqueue_t* queue = NULL;
      bool success = false;
      int full_count = 0;                                             
      //for (size_t enqueue_slot = 0, i = 0; ;)
      for (size_t i = 0; ;)
      {
        //i = ++enqueue_slot & (QUEUE_COUNT - 1);
        i = enqueue_slot_.fetch_add(1, std::memory_order_relaxed) & (QUEUE_COUNT - 1);
        queue = queues_[i].queue.exchange(nullptr, std::memory_order_acquire);    
        if (queue) {
          success = queue->push(v);                                               
          queues_[i].queue.store(queue, std::memory_order_release);               
          if (success) return success;                                            
          if (++full_count == QUEUE_COUNT) break;                                 
        }
        static const struct timespec ns = { 0, 1 };
        nanosleep(&ns, NULL);
      };
      count_.fetch_add(-1);
      return success;
    }

  void reset() { 
    count_ = enqueue_slot_ = dequeue_slot_ = 0;
    for (int i = 0; i < QUEUE_COUNT; ++i) {
      subqueue_t* queue = queues_[i].queue;
      queue->reset();
    }
  }

  private:
    enum { QUEUE_COUNT = 16 };
    subqueue_ptr_t queues_[QUEUE_COUNT];
    std::atomic<int> count_;
    std::atomic<size_t> enqueue_slot_;
    std::atomic<size_t> dequeue_slot_;
};

template <typename T> class concurrent_std_queue {
  using subqueue_t = std::queue<T>;
  struct subqueue_ptr_t {
    subqueue_ptr_t() : queue() {}
    std::atomic<subqueue_t*> queue;
    char padding[64 - sizeof(queue)]; // Wypełnienie do wiersza bufora.
  };

  public:
    explicit concurrent_std_queue() {
      for (int i = 0; i < QUEUE_COUNT; ++i) {
        queues_[i].queue.store(new subqueue_t, std::memory_order_relaxed);
      }
    }
    ~concurrent_std_queue() {
      for (int i = 0; i < QUEUE_COUNT; ++i) {
        subqueue_t* queue = queues_[i].queue.exchange(nullptr, std::memory_order_relaxed);
        delete queue;
      }
    }

    // Ile pozycji jest w kolejce?
    size_t size() const { return count_.load(std::memory_order_acquire); }

    // Czy kolejka jest pusta?
    bool empty() const { return size() == 0; }

    // Uzyskanie elementu z kolejki.
    // Metoda ta powoduje blokowanie albo do chwili opróżnienia kolejki (size() == 0) albo
    // zwrócenia elementu z jednej z podkolejek.
    std::optional<T> pop() {
      std::optional<T> res;
      if (count_.load(std::memory_order_acquire) == 0) {
        return res;
      }
      // Przejęcie własności podkolejki. Dla wskaźnika podkolejki jest ponownie ustawiana
	  // wartość NULL, gdy wątek wywołujący jest w posiadaniu podkolejki. Po zrealizowaniu
	  // działania przywracanie wskaźnika powoduje wycofanie własności. Dostępna podkolejka
	  // może być pusta, ale nie oznacza to, że nie ma żadnych pozycji. Konieczne jest
      // sprawdzenie innych kolejek. Pętla może zostać zakończona po uzyskaniu elementu
	  // albo wtedy, gdy licznik elementów wskazuje na brak pozycji.
	  // Zauważ, że dekrementacja licznika nie jest atomowa w przypadku usuwania pozycji
	  // z kolejki, dlatego przez krótki czas blokada wirująca może być używana dla pustych
	  // kolejek do momentu wyrównania wartości licznika.
      subqueue_t* queue = NULL;
      bool success = false;
      for (size_t i = 0; ;)
      {
        i = dequeue_slot_.fetch_add(1, std::memory_order_relaxed) & (QUEUE_COUNT - 1);
        queue = queues_[i].queue.exchange(nullptr, std::memory_order_acquire);  // Przejmowanie własności podkolejki
        if (queue) {
          if (!queue->empty()) {                                                // Usuwanie elementu z kolejki po przejęciu do niej 
		                                                                        // prawa własności
            success = true;
            res.emplace(std::move(queue->front()));
            queue->pop();
            queues_[i].queue.store(queue, std::memory_order_release);           // Wycofanie prawa własności
            break;                                                              // Uzyskanie elementu
          } else {                                                              // Podkolejka jest pusta. Spróbuj użyć innej.
            queues_[i].queue.store(queue, std::memory_order_release);           // Wycofanie prawa własności
            if (count_.load(std::memory_order_acquire) == 0) goto EMPTY;        // Brak elementu i brak kolejnych
          }
        } else {                                                                // Kolejka ma wartość NULL. Nic nie musi być wycofywane
          if (count_.load(std::memory_order_acquire) == 0) goto EMPTY;          // Nie uzyskano dostępu do kolejki, ale nie ma już żadnych
                                                                                // pozycji
        }
        if (success) break;
        if (count_.load(std::memory_order_acquire) == 0) break; 
        static const struct timespec ns = { 0, 1 };
        nanosleep(&ns, NULL);
      };
      // Jeśli istnieje element, zmniejszana jest wartość licznika elementów w kolejce.
      count_.fetch_add(-1);
EMPTY:
      return res;
  }

  bool push(const T& v) {
    count_.fetch_add(1);
    subqueue_t* queue = NULL;
    bool success = false;
    for (size_t i = 0; ;)
    {
      i = enqueue_slot_.fetch_add(1, std::memory_order_relaxed) & (QUEUE_COUNT - 1);
      queue = queues_[i].queue.exchange(nullptr, std::memory_order_acquire);    // Przejmowanie własności podkolejki 
      if (queue) {
        success = true;
        queue->push(v);                                                         // Dodawanie elementu do kolejki po przejęciu do niej 
		                                                                        // prawa własności 
        queues_[i].queue.store(queue, std::memory_order_release);               // Wycofanie prawa własności
        return success;
      }
      static const struct timespec ns = { 0, 1 };
      nanosleep(&ns, NULL);
    };
    return success;
  }

  void reset() { 
    count_ = enqueue_slot_ = dequeue_slot_ = 0;
    for (int i = 0; i < QUEUE_COUNT; ++i) {
      subqueue_t* queue = queues_[i].queue;
      subqueue_t().swap(*queue);
    }
  }

  private:
  enum { QUEUE_COUNT = 16 };
  subqueue_ptr_t queues_[QUEUE_COUNT];
  std::atomic<int> count_;
  std::atomic<size_t> enqueue_slot_;
  std::atomic<size_t> dequeue_slot_;
};

concurrent_std_queue<int> q;
concurrent_std_queue<LS> lq;

void BM_queue(benchmark::State& state) {
  if (state.thread_index == 0) q.reset();
  const size_t N = state.range(0);
  for (auto _ : state) {
    for (size_t i = 0; i < N; ++i) q.push(i);
    for (size_t i = 0; i < N; ++i) benchmark::DoNotOptimize(q.pop());
  }
  state.SetItemsProcessed(state.iterations()*N);
}

void BM_queue_prod_cons(benchmark::State& state) {
  if ((state.threads & 1) == 1) state.SkipWithError("Niezbędna parzysta liczba wątków!");
  if (state.thread_index == 0) q.reset();
  const bool producer = state.thread_index & 1;
  const size_t N = state.range(0);
  for (auto _ : state) {
    if (producer) {
      for (size_t i = 0; i < N; ++i) q.push(i);
    } else {
      for (size_t i = 0; i < N; ++i) benchmark::DoNotOptimize(q.pop());
    }
  }
  state.SetItemsProcessed(state.iterations()*N);
}

void BM_lqueue_prod_cons(benchmark::State& state) {
  if ((state.threads & 1) == 1) state.SkipWithError("Niezbędna parzysta liczba wątków!");
  if (state.thread_index == 0) lq.reset();
  const bool producer = state.thread_index & 1;
  const size_t N = state.range(0);
  for (auto _ : state) {
    if (producer) {
      for (size_t i = 0; i < N; ++i) lq.push(i);
    } else {
      for (size_t i = 0; i < N; ++i) benchmark::DoNotOptimize(lq.pop());
    }
  }
  state.SetItemsProcessed(state.iterations()*N);
}

static const long numcpu = sysconf(_SC_NPROCESSORS_CONF);

#define ARGS \
  ->Arg(1) \
  ->ThreadRange(1, numcpu) \
  ->UseRealTime()

BENCHMARK(BM_queue) ARGS;
BENCHMARK(BM_queue_prod_cons) ARGS;
BENCHMARK(BM_lqueue_prod_cons) ARGS;

BENCHMARK_MAIN();
