// Copyright 2019 The TensorFlow Authors. All Rights Reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
// =============================================================================

/**
 * Klasa opakowująca funkcje obsługi kamery internetowej w celu przechwytywania danych Tensor4D.
 */


class Webcam {
  /**
   * @param {HTMLVideoElement} webcamElement Obiekt typu HTMLVideoElement
   *      reprezentujący dane z kamery.
   */
  constructor(webcamElement) {
    this.webcamElement = webcamElement;
  }

  /**
	* Przechwytuje pojedynczą klatkę z kamery internetowej i normalizuje dane w zakresie od -1 do 1.
   * Zwraca obraz (paczkę 1-elementową) o kształcie [1, w, h, c].
   */


  capture() {
    return tf.tidy(() => {
		// Odczytuje obraz jako typ Tensor z elementu <video> kamery internetowej.
      const webcamImage = tf.browser.fromPixels(this.webcamElement);

      const reversedImage = webcamImage.reverse(1);

		// Wykorzystuje środkowy, kwadratowy obszar z prostokątnego obrazu kamery internetowej.
      const croppedImage = this.cropImage(reversedImage);

		// Dodaje wymiar, aby uzyskać rozmiar partii równy 1.
      const batchedImage = croppedImage.expandDims(0);

		// Normalizuje obraz, aby składał się z wartości zmiennoprzecinkowych z zakresu od -1 do 1.
		// Pierwotny obraz składa się z pikseli o wartościach 0-255, które dzielimy przez 127,
		// a następnie od wyniku odejmujemy 1.
      return batchedImage.toFloat().div(tf.scalar(127)).sub(tf.scalar(1));
    });
  }

  /**
   * Przycina tensor obrazu, aby uzyskać obszar o kształcie kwadratu.
   * @param {Tensor4D} img Tensor obrazu wejściowego.
   */


  cropImage(img) {
    const size = Math.min(img.shape[0], img.shape[1]);
    const centerHeight = img.shape[0] / 2;
    const beginHeight = centerHeight - (size / 2);
    const centerWidth = img.shape[1] / 2;
    const beginWidth = centerWidth - (size / 2);
    return img.slice([beginHeight, beginWidth, 0], [size, size, 3]);
  }

  /**
	* Dostosowuje rozmiar wideo w celu wycięcia środkowego fragmentu obrazu w kształcie kwadratu.
   * @param {number} width Szerokość elementu wideo.
   * @param {number} height Wysokość elementu wideo.
   */


  adjustVideoSize(width, height) {
    const aspectRatio = width / height;
    if (width >= height) {
      this.webcamElement.width = aspectRatio * this.webcamElement.height;
    } else if (width < height) {
      this.webcamElement.height = this.webcamElement.width / aspectRatio;
    }
  }

  async setup() {
    return new Promise((resolve, reject) => {
      navigator.getUserMedia = navigator.getUserMedia ||
          navigator.webkitGetUserMedia || navigator.mozGetUserMedia ||
          navigator.msGetUserMedia;
      if (navigator.getUserMedia) {
        navigator.getUserMedia(
            {video: {width: 224, height: 224}},
            stream => {
              this.webcamElement.srcObject = stream;
              this.webcamElement.addEventListener('loadeddata', async () => {
                this.adjustVideoSize(
                    this.webcamElement.videoWidth,
                    this.webcamElement.videoHeight);
                resolve();
              }, false);
            },
            error => {
              reject(error);
            });
      } else {
        reject();
      }
    });
  }
}
