/*
 * hardfaults.c
 *
 * Ten plik demonstruje kilka "twardych" błędów.
 *
 * Są to proste błędy, wynikające z oczywistych pomyłek. Jedna większość twardych błędów
 * nie jest tak oczywista.
 *
 * Niektóre przykłady są zaczerpnięte z artykułu dostępnego pod adresem
 * https://mcuoneclipse.com/2012/11/24/debugging-hard-faults-on-arm-cortex-m/
 * Omawia on również debugowanie błędów twardych na procesorach Cortex-M,
 * choć nie są to procesory STM.
 *
 * Innym dobrym źródłem informacji o debugowaniu twardych błędów jest strona
 * https://www.nathantsoi.com/blog/stm32-hardfault-debugging/index.html,
 * która traktuje o procesorach STM
 *
 * Najlepszym zasobem do analizowania i debugowania twardych błędów jest strona
 * https://interrupt.memfault.com/blog/cortex-m-fault-debug
 * Wymaga ona wiersza poleceń GDB, który jest dostępny w konsoli debugera.
 *
 * Arm udostępnia dokumentację... całkiem dobrej jakości
 * https://developer.arm.com/documentation/dui0552/a/cortex-m3-peripherals/system-control-block/configurable-fault-status-register
 *
 */

#include <stdint.h>
#include <stdlib.h>
#include "stm32l4xx.h"

int divide_by_zero(void)
{
	int a = 1;
	int c = 0;
	int b = a/c;
	return b; // wymusza rzeczywiste wykonanie kodu przez kompilator
}

 // niezainicjalizowana zmienna globalna zostaje zainicjalizowana na 0
int* global_ptr_to_null = NULL;
int* global_ptr_unitialized;
int write_to_null(void) {
  int* ptr_to_null = NULL;
  int* ptr_unitialized;

  *global_ptr_to_null  = 10;    /* próbuje zapisać adres zero */
  *global_ptr_unitialized = 10; /*  próbuje zapisać adres zero */
  *ptr_to_null  = 10;           /*  próbuje zapisać adres zero */
  *ptr_unitialized = 10;    		/*  próbuje zapisać ?? gdzieś ?? */

  return *global_ptr_to_null + *global_ptr_unitialized +
          *ptr_to_null + *ptr_unitialized;
}
// UWAGA: Może będziesz musiał ustawić MPU w tryb ochrony wskaźników
// Patrz: https://interrupt.memfault.com/blog/fix-bugs-and-secure-firmware-with-the-mpu#enable-memmanage-fault-handler

// Zakładamy, że 0xE0000000 jest nieprawidłową instrukcją
int illegal_instruction_execution(void) {
  int instruction = 0xE0000000;
  int (*bad_instruction)(void) = (int(*)())(&instruction);
  return bad_instruction();
}

// podobnie jak niżej, gdzie wykonywane jest to, co akurat znajduje się pod danym adresem
int illegal_address_execution(void) {
  int (*illegal_address)(void) = (int(*)())0xE0000000;
  return illegal_address();
}

void (*fun_ptr)(void);	// zmienna globalna o domyślnej wartości zero
void call_null_pointer_function(void)
{
  fun_ptr(); /* wykona kod pod adresem zero, często kod resetowania */
}

/******************************************************************************************************
 * Zwracanie pamięci:
 * Zwracanie pamięci stosu może prowadzić do błędów stosu i awarii
 * Zwracanie zwolnionej pamięci zwykle prowadzi tylko do błędów w danych
*******************************************************************************************************/
int* dont_return_stack_memory(void) {
	int stack_memory[100];
	return stack_memory;
}

int* dont_return_malloc_and_freed_memory(void) {
	int* memory = malloc(100);
	free(memory);
	return memory;
}

int this_is_ok(void) {
	int a = 0;
	int b = a+1;
	return b;
}

/******************************************************************************************************
* W procesorze STM32 Cortex-M4 zmienna smallVariable ma wartość 0xdd, ale zależy to
* od procesora i jego kolejności bajtów.
*******************************************************************************************************/
uint8_t unaligned_access_ok(void)
{
	uint32_t bigVariable = 0xAABBCCDD;
	uint32_t  *ptr = &bigVariable;
	uint8_t  smallVariable = *ptr;
	return smallVariable;
}

/******************************************************************************************************
 * Niewyrówany dostęp jest automatycznie błędem twardym w CM0.
 *
 * M4 pozwoli nam wykonać ten kod... ale jeśli chcemy wywołać twardy błąd,
 * aby uniknąć spowolnienia, musimy ustawić CCR.UNALIGN_TRP
 * albo użyć flagi kompilacji mno-unaligned-access
 *
 * W procesorze STM32 Cortex-M4 zmienna val_BB_to_EE jest równa 0xeeddccbb,
 * ale zależy to od procesora i jego kolejności bajtów.
*******************************************************************************************************/
 uint32_t unaligned_access_bad(int index)
{
	uint8_t buffer[6] = {0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF};
	uint8_t i = index;
    uint32_t val_BB_to_EE = *((uint32_t *)( &buffer[i] ));
    return val_BB_to_EE;
}

// W kompilatorze ARM GCC
// -munaligned-access jest opcją domyślną; kod powinien
// sięgnąć do bufora i wydobyć niewyrównaną zmienną w kilku poleceniach asemblera
// Jednak opcja -mno-unaligned-access powinna prowadzić do dłuższego kodu, zawierającego
// więcej instrukcji w celu uniknięcia niewyrównanego dostępu do pamięci


void do_some_hardfaults(void)
{
	divide_by_zero(); 			  	// przykład z rejestrami
	write_to_null();				// null i zmienna niezainicjalizowana: MPU nie ma z tym problemu

	illegal_instruction_execution(); // przykład budowania procedury obsługi

    call_null_pointer_function();    // wpis na blogu memfault: https://interrupt.memfault.com/blog/cortex-m-fault-debug

    /* UNALIGNED ACCESS */
	unaligned_access_bad(1);
	unaligned_access_ok();
	SCB->CCR |= (1<<3);			  	// włącza twardy błąd, który nie zezwala na niewyrównany dostęp
	unaligned_access_ok();			// działa
	unaligned_access_bad(0);		// działa
	unaligned_access_bad(1);		// twardy błąd


}


/******************************************************************************************************
 * Poniższy kod pomaga zrozumieć błąd twardy na podstawie rejestrów procesora
 * W mojej klasie będę używać tego kodu do debugowania powyższych twardych błędów
*******************************************************************************************************/

// Początkowo bład trafia do pustej funkcji void HardFault_Handler(void) w pliku stm32f4xx_it.c
// Połączenie to jest skonfigurowane w pliku Startup/startup_stm32f429zitx.s
// W funkcji tej możemy obejrzeć stos wywołań i wyśledzić błąd w naszym kodzie
// W rejestrach SFRs możemy przyjrzeć się wartościom Cortex_M4 -> Control -> CFSR
// oraz Control -> CCR Control SHCSR

// Funkcja ta jednak nie robi zbyt wiele, więc dodajmy do niej coś bardziej użytecznego


// Na podstawie https://www.freertos.org/Debugging-Hard-Faults-On-Cortex-M-Microcontrollers.html

// #define  NEW_HANLDER_1
#ifdef  NEW_HANLDER_1

void HardFault_Handler(void)
{
	__asm volatile
	(
		" movs r0,#4    \n"
		" movs r1, lr   \n"
		" tst r0, r1    \n"
		" beq _MSP		\n"
		" mrs r0, psp	\n"
		" b _HALT		\n"
		"_MSP:			\n"
		" mrs r0, msp	\n"
		"_HALT:			\n"
		" ldr r1,[r0,#20] \n"
		"  b hard_fault_handler_c \n"
		"  bkpt #0 \n"
	  );
}
void hard_fault_handler_c(unsigned long *hardfault_args){
  volatile unsigned long stacked_r0 ;
  volatile unsigned long stacked_r1 ;
  volatile unsigned long stacked_r2 ;
  volatile unsigned long stacked_r3 ;
  volatile unsigned long stacked_r12 ;
  volatile unsigned long stacked_lr ;
  volatile unsigned long stacked_pc ;
  volatile unsigned long stacked_psr ;
  volatile unsigned long _CFSR ;
  volatile unsigned long _HFSR ;
  volatile unsigned long _DFSR ;
  volatile unsigned long _AFSR ;
  volatile unsigned long _BFAR ;
  volatile unsigned long _MMAR ;

  stacked_r0 = ((unsigned long)hardfault_args[0]) ;
  stacked_r1 = ((unsigned long)hardfault_args[1]) ;
  stacked_r2 = ((unsigned long)hardfault_args[2]) ;
  stacked_r3 = ((unsigned long)hardfault_args[3]) ;
  stacked_r12 = ((unsigned long)hardfault_args[4]) ;
  stacked_lr = ((unsigned long)hardfault_args[5]) ;
  stacked_pc = ((unsigned long)hardfault_args[6]) ;
  stacked_psr = ((unsigned long)hardfault_args[7]) ;

  // Configurable Fault Status Register
  //	 podsumowanie błędu lub błędów, które spowodowały wyjątek
  // Zawiera:
  // * UsageFault Status Register (UFSR)- błędy, które nie mają związku z dostępem do pamięci
  // * BusFault Status Register (BFSR) - błędy związaze ze wstępnym pobieraniem instrukcji albo nieudanym dostępem do pamięci
  // * MemManage Status Register (MMFSR) - błędy jednostki ochrony pamięci (Memory Protection Unit)
  _CFSR = (*((volatile unsigned long *)(0xE000ED28))) ;

  // Hard Fault Status Register - bardzo ograniczony, wyjaśnia przyczynę wyzwolenia błędu:
  // zdarzenie debugowania, skonfigurowane zdarzenie albo błąd w tablicy wektorów
  _HFSR = (*((volatile unsigned long *)(0xE000ED2C))) ;

  // Debug Fault Status Register
  _DFSR = (*((volatile unsigned long *)(0xE000ED30))) ;

  // Auxiliary Fault Status Register
  _AFSR = (*((volatile unsigned long *)(0xE000ED3C))) ;

  // Odczytujemy rejestry adresu błędu (Fault Address Registers).
  // Mogą one nie zawierać prawidłowych wartości.
  // Można sprawdzić flagi BFARVALID/MMARVALID, aby dowiedzieć się, czy wartości są prawidłowe
  // MemManage Fault Address Register
  _MMAR = (*((volatile unsigned long *)(0xE000ED34))) ;
  // Bus Fault Address Register
  _BFAR = (*((volatile unsigned long *)(0xE000ED38))) ;

  __asm("BKPT #0\n") ; // Przerwanie do debugera
}
#endif // NEW_HANDLER_1

// #define NEW_HANDLER_MEMFAULT
#ifdef NEW_HANDLER_MEMFAULT

typedef struct __attribute__((packed)) ContextStateFrame {
  uint32_t r0;
  uint32_t r1;
  uint32_t r2;
  uint32_t r3;
  uint32_t r12;
  uint32_t lr;
  uint32_t return_address;
  uint32_t xpsr;
} sContextStateFrame;

#define HARDFAULT_HANDLING_ASM(_x)               \
  __asm volatile(                                \
      "tst lr, #4 \n"                            \
      "ite eq \n"                                \
      "mrseq r0, msp \n"                         \
      "mrsne r0, psp \n"                         \
      "b my_fault_handler_c \n"                  \
)

void HardFault_Handler(void)
{
	HARDFAULT_HANDLING_ASM();
}

// Dodajemy do pamięci RAM sekcję zrzutu rdzenia, która będzie 
// umieszczona w konkretnej lokalizacji, więc po ponownym uruchomieniu
// będziemy mogli do niej zajrzeć i sprawdzić, czy jest
// w niej coś użytecznego
//  .CoreDump :
//  {
 // } > RAM2
#define COREDUMP_KEY 0xE0C2024
__attribute__((packed)) struct sCoreDump {
uint32_t key; // musi być równy COREDUMP_KEY
uint32_t cause;
uint32_t r0;
uint32_t r1;
uint32_t r2;
uint32_t r3;
uint32_t returnAddress;
uint32_t stackPointer;
int32_t lastBattReading;
} __attribute__((section(".CoreDump"))) coreDump;

// Wyłączamy optymalizację tej funkcji, aby optymalizator
// nie usunął argumentu "frame"
__attribute__((optimize("O0")))

void my_fault_handler_c(sContextStateFrame *frame)
{
    coreDump.key = COREDUMP_KEY;
    coreDump.r0 = frame->r0;
    coreDump.r1 = frame->r1;
    coreDump.r2 = frame->r2;
    coreDump.r3 = frame->r3;
    coreDump.returnAddress = frame->return_address;
    coreDump.stackPointer = frame->xpsr;
    coreDump.lastBattReading = 0; // pobieramy ze zmiennej, nie przez wykonanie kodu

// Jeśli dołączony jest debuger, wykonujemy instrukcję
// punktu przerwania, żebyśmy mogli zobaczyć, co wywołało błąd
  __asm("BKPT #0\n") ; // Przerwanie do debugera

  // Logika obsługi wyjątku. Zwykle:
  //  - rejestrujemy błąd do późniejszej analizy
  //  - Jeśli da się usunąć skutki błędu
  //    - zerujemy błędy i wracamy
  //  - w przeciwnym razie
  //    - ponownie uruchamiamy system
}

#endif // NEW_HANDLER_MEMFAULT
