package com.reactivedesignpatterns.chapter11

import java.util.concurrent.TimeoutException

import akka.actor.{Actor, ActorRef, Props}
import akka.util.Timeout
import com.reactivedesignpatterns.Defaults._

import scala.concurrent.duration._
import scala.concurrent.{ExecutionContext, Future}

class TranslationService {
  import ExecutionContext.Implicits.global

  def translate(input: String): Future[String] = Future { Thread.sleep(100); "Jak się masz?" }

  def translate(input: String, ec: ExecutionContext): Future[String] =
    Future { Thread.sleep(100); "Jak się masz?" }(ec)
}

/**
 * Inna implementacja oparta na aktorach, zastosowana w przykładzie
 * pokazującym jak wykryć brak niejawnie wywołanych komunikatów za
 * pomocą testu adaptera protokołu w usłudze tłumaczącej.
 */
object TranslationService {

  /**
   * Najprostsza wersja 1 protokołu. Odpowiedź jest po prostu typu String.
   */
  case class TranslateV1(query: String, replyTo: ActorRef)

  /**
   * Impementacja protokołu TranslateV1.
   */
  private class TranslatorV1 extends Actor {
    def receive = {
      case TranslateV1(query, replyTo) =>
        if (query == "en:pl:How are you?") {
          replyTo ! "Jak się masz?"
        } else {
          replyTo ! s"błąd: nie można przetłumaczyć '$query'"
        }
    }
  }
  
  def propsV1: Props = Props(new TranslatorV1)

  /**
   * Bardziej zaawansowana wersja 2 protokołu z odpowiednimi typami
   * odpowiedzi. Języki dla uproszczenia są oznaczane ciągami znaków.
   * W docelowym projekcie należy je modelować jako odpowiednie typy
   * Language (statyczna enumeracja oparta na rejestracji wartości
   * podczas działania programu).
   */
  case class TranslateV2(phrase: String, inputLanguage: String, outputLanguage: String, replyTo: ActorRef)

  sealed trait TranslationResponseV2
  case class TranslationV2(inputPhrase: String, outputPhrase: String, inputLanguage: String, outputLanguage: String)
  case class TranslationErrorV2(inputPhrase: String, inputLanguage: String, outputLanguage: String, errorMessage: String)

  /**
   * Implementacja protokołu TranslateV2 oparta na TranslatorV1.
   */
  private class TranslatorV2(v1: ActorRef) extends Actor {
    implicit val timeout = Timeout(5.seconds)
    import context.dispatcher

    def receive = {
      case TranslateV2(phrase, in, out, replyTo) =>
        v1 ? (TranslateV1(s"$in:$out:$phrase", _)) collect {
          case str: String =>
            if (str.startsWith("error:")) {
              TranslationErrorV2(phrase, in, out, str.substring(6))
            } else {
              TranslationV2(phrase, str, in, out)
            }
        } recover {
          case _: TimeoutException =>
            TranslationErrorV2(phrase, in, out, "timeout podczas komunikacji z back-end V1")
        } pipeTo replyTo
    }
  }
  
  def propsV2(v1: ActorRef): Props = Props(new TranslatorV2(v1))
}
