#include <string>
#include <vector>
#include <iostream>
#include <mutex>
using namespace std;
#include <boost/any.hpp>
using namespace boost;
#include <algorithm>

namespace {

struct Person;

struct PersonListener
{
  virtual ~PersonListener() = default;
  virtual void person_changed(Person& p, const string& property_name, const any new_value) = 0;
};

static mutex mtx;

struct Person
{
  explicit Person(int age)
    : age(age)
  {
  }

  int get_age() const
  {
    return age;
  }

  void set_age(const int age)
  {
    if (this->age == age) return;

    auto old_c_v = get_can_vote();

    this->age = age;
    notify("age", this->age);

    auto new_c_v = get_can_vote();
    if (old_c_v != new_c_v)
    {
      notify("can_vote", new_c_v);
    }
  }

  bool get_can_vote() const
  {
    return age >= 18;
  }

  void subscribe(PersonListener* pl)
  {
    lock_guard<mutex> guard{ mtx };

    // zabezpieczy si przed istniejcymi ju referencjami?
    if (find(begin(listeners), end(listeners), pl) == end(listeners))
      listeners.push_back(pl);
  }
  void ubsubscribe(PersonListener* pl)
  {
    lock_guard<mutex> guard{ mtx };
    if (listeners.empty()) return;
    // wiele identycznych odbiornikw?
    // idiom erase-remove?
    for (auto it = listeners.begin(); it != listeners.end(); ++it)
    {
      if (*it == pl)
      {
        *it = nullptr; // po prostu oznacz jako nullptr
      }
    }
  }
private:
  int age;
  vector<PersonListener*> listeners;

  void notify(const string& property_name, const any new_value)
  {
    lock_guard<mutex> guard{ mtx };
    for (const auto listener : listeners)
      if (listener)
        listener->person_changed(*this, property_name, new_value);

    // erase-remove 
    listeners.erase(
      remove(listeners.begin(), listeners.end(), nullptr),
      listeners.end()
    );
  }

  // std::list (atwiej usuwa elementy)
  // concurrent_vector? tak, ale nie da si atwo uy idiomu erase-remove
};

struct ConsoleListener : PersonListener
{
  void person_changed(Person& p, const string& property_name, 
    const any new_value) override
  {
    cout << "Waciwo " << property_name << " obiektu Person ma teraz warto ";
    if (property_name == "age")
    {
      cout << any_cast<int>(new_value);
    }
    else if (property_name == "can_vote")
    {
      cout << any_cast<bool>(new_value);
    }
    cout << "\n";
  }
};

int main__x_()
{
  Person p{ 16 };
  ConsoleListener cl;
  p.subscribe(&cl);
  p.subscribe(&cl); // ignorowane
  p.set_age(17);
  p.set_age(18);
  p.ubsubscribe(&cl);
  p.set_age(19);


  getchar();
  return 0;
}

}