/*---------------------------------------------------------------------------*/
    #include <errno.h>
    #include <fcntl.h>
    #include <stdarg.h>
    #include <stdio.h>
    #include <stdlib.h>
    #include <sys/stat.h>
    #include <sys/types.h>
    #include <unistd.h>

    #include <bfd.h>
    #include <dis-asm.h>


    struct ASM_INSN {
        char mnemonic[16];
        char src[32];
        char dest[32];
        char arg[32];
    } curr_insn;

    int disprintf(FILE *stream, const char *format, ...){
        /* Zastpienie funkcji fprintf( ).
         * UWAGA: poniej zaoono porzdek src, dest z deasemblera */
        va_list args;
        char *str;

        va_start (args, format);
        str = va_arg(args, char*);

        /* fatalnie, libopcodes przekazuje w wywoaniu one mnemonik lub operand
           i przekazuje src dwukrotnie */
        if ( ! curr_insn.mnemonic[0] ) {
            strncpy(curr_insn.mnemonic, str, 15);
        } else if ( ! curr_insn.src[0] ) {
            strncpy(curr_insn.src, str, 31);
        } else if ( ! curr_insn.dest[0] ) {
            strncpy(curr_insn.dest, str, 31);
            if (strncmp(curr_insn.dest, "DN", 2) == 0)
                    curr_insn.dest[0] = 0;
        } else {
            if ( ! strcmp( curr_insn.src, curr_insn.dest ) ) {
                /* src zosta przekazany dwukrotnie */
                strncpy(curr_insn.dest, str, 31);
            } else {
                strncpy(curr_insn.arg, str, 31);
            }
        }
        va_end (args);

        return(0);
    }

    void print_insn( void ) {
        printf("\t%s", curr_insn.mnemonic);
        if ( curr_insn.src[0] ) {
            printf("\t%s", curr_insn.src );
            if ( curr_insn.dest[0] ) {
                printf(", %s", curr_insn.dest );
                if ( curr_insn.arg[0] ) {
                    printf(", %s", curr_insn.arg );
                }
            }
        }
        return;
    }

    int disassemble_forward( disassembler_ftype disassemble_fn,
                    disassemble_info *info, unsigned long rva ) {
        int bytes = 0;
    
        while ( bytes < info->buffer_length ) {
            /* wywoanie deasemblera libopcodes */
            memset( &curr_insn, 0, sizeof( struct ASM_INSN ));
            bytes += (*disassemble_fn)(info->buffer_vma + bytes, info); 
    
            /* -- wywietla nazwy symboli jako etykiety -- */
            /* wywietla adres instrukcji */
            printf("%8X : ", info->buffer_vma + bytes);
            /* -- analiza deasemblowanej instrukcji -- */
            print_insn( );
            printf("\n");
        }
        return( bytes );
    }

    int disassemble_buffer( disassembler_ftype disassemble_fn, 
                    disassemble_info *info ) {
        int i, size, bytes = 0;
    
        while ( bytes < info->buffer_length ) {
            /* wywoanie deasemblera libopcodes */
            memset( &curr_insn, 0, sizeof( struct ASM_INSN ));
            size = (*disassemble_fn)(info->buffer_vma + bytes, info); 
    
            /* -- analiza zdeasemblowanych instrukcji -- */
    
            /* -- wywietla nazwy symboli jako etykiety -- */
            printf("%8X:   ", info->buffer_vma + bytes);
            /* wywietla bajty szesnastkowo */
            for ( i = 0; i < 8; i++ ) {
                if ( i < size )     
                    printf("%02X ", info->buffer[bytes + i]);
                else 
                    printf("   ");
            }
            print_insn( );
            printf("\n");
            bytes += size;    /* przeniesienie pozycji w buforze */
        }
        return( bytes );
    }

    static void disassemble( bfd *b, asection *s, unsigned char *buf, 
                        int size, unsigned long buf_vma ) {
        disassembler_ftype disassemble_fn;
        static disassemble_info info = {0};
    
        if ( ! buf )     return;
        if ( ! info.arch ) {
            /* inicjacja wszystkiego */
            INIT_DISASSEMBLE_INFO(info, stdout, disprintf);
            info.arch = bfd_get_arch(b);
            info.mach = bfd_mach_i386_i386;    /* BFD_guess nie dziaa */
            info.flavour = bfd_get_flavour(b);
            info.endian = b->xvec->byteorder;
            /* mona to zastpi wasnymi procedurami 
              info.read_memory_func = buffer_read_memory;
              info.memory_error_func = perror_memory;
              info.print_address_func = generic_print_address;
              info.symbol_at_address_func = generic_symbol_at_address;
              info.fprintf_func = disprintf; //obsugiwane w powyszym makro
              info.stream = stdout;          //jak wyej
              info.symbols = NULL;
              info.num_symbols = 0;
            */
            info.display_endian = BFD_ENDIAN_LITTLE;
    
        }
    
        /* wybr funkcji deasemblera */
        disassemble_fn = print_insn_i386_att;
        /* disassemble_fn = print_insn_i386_intel; */
        /* ponisze zale od sekcji */
        info.section = s;            /* sekcja do deasemblacji */
        info.buffer = buf;        /* bufor bajtw do deasemblacji */
        info.buffer_length = size;    /* rozmiar bufora */
        info.buffer_vma = buf_vma;    /* podstawa RVA bufora */
        
        disassemble_buffer( disassemble_fn, &info );
    
        return;
    }

    static void print_section_header( asection *s, const char *mode ) {
    
        printf("Deasemblacja sekcji %s jako %s\n", s->name, mode );
        printf("RVA: %08X LMA: %08X Znaczniki: %08X Rozmiar: %X\n", s->vma,
                s->lma, s->flags, s->_cooked_size );
        printf( "---------------------------------------------------------"
              "-----------------------\n" );
        return;
    }

    static void disasm_section_code( bfd *b, asection *section ) {
        int size;
        unsigned char *buf;

        size = bfd_section_size( b, section );
        buf = calloc( size, 1 );
        if (! buf || ! bfd_get_section_contents(b, section, buf, 0, size ))
            return;
        print_section_header( section, "code" );
        disassemble( b, section, buf, size, section->vma );
        printf("\n\n");
        free( buf );
        return;
    }

    static void disasm_section_data( bfd *b, asection *section ) {
        int i, j, size;
        unsigned char *buf;

        size = bfd_section_size( b, section );
        buf = calloc( size, 1 );
        if ( ! bfd_get_section_contents( b, section, buf, 0, size ) )
            return;
        print_section_header( section, "data" );
        /* wywietlenie danych w postaci szesnastkowej */
        for ( i = 0; i < size; i +=16 ) {
            printf( "%08X:   ", section->vma + i );
            for (j = 0; j < 16 && j+i < size; j++ )  
                printf("%02X ", buf[i+j] );
            for ( ; j < 16; j++ )
                printf("   ");
            printf("  ");
            for (j = 0; j < 16 && j+i < size; j++ ) 
                printf("%c", isprint(buf[i+j]) ? buf[i+j] : '.' );
            printf("\n");
        }
        printf("\n\n");
        free( buf );
        return;
    }

    static void disasm_section( bfd *b, asection *section, PTR data ){
        if ( ! section->flags & SEC_ALLOC )  return;
        if ( ! section->flags & SEC_LOAD ) return;
        if ( section->flags & SEC_LINKER_CREATED ) return;
        if ( section->flags & SEC_CODE) {
            if ( ! strncmp(".plt", section->name, 4) ||
                 ! strncmp(".got", section->name, 4) ) {
                return;
            }
            disasm_section_code( b, section );
        } else if ( (section->flags & SEC_DATA ||
                 section->flags & SEC_READONLY) &&
                section->flags & SEC_HAS_CONTENTS  ) {
            disasm_section_data( b, section );
        }
        return;
    }

    int main( int argc, char **argv ) {
              struct stat s;
        bfd *infile;

          if ( argc < 2 ) {
                fprintf(stderr, "Uycie: %s nazwa_pliku\n", argv[0]);
                return(1);
          }
          if ( stat( argv[1], &s) ) {
                fprintf(stderr, "Bd: %s\n", strerror(errno) );
                return(2);
          }

        bfd_init( );

        /* otwarcie pliku wejciowego do odczytu */
        infile = bfd_openr( argv[1], NULL );
        if ( ! infile ) {
            bfd_perror( "Bd w pliku wejciowym" );
            return(3);
        }

        if ( bfd_check_format (infile, bfd_object ) || 
              bfd_check_format(infile, bfd_archive ) ) {
            bfd_map_over_sections( infile, disasm_section, NULL );
        } else  {
            fprintf( stderr, "Bd: nieznany format pliku\n");
            return(5);
        }

        bfd_close(infile);
    
          return(0);
    }
/*-----------------------------------------------------------------------*/
