package com.apress.springrecipes.jbpm.jbpm4.customers;

import org.apache.commons.lang.exception.ExceptionUtils;

import org.apache.log4j.Logger;

import org.jbpm.api.*;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import org.springframework.core.io.ClassPathResource;

import org.springframework.orm.hibernate3.HibernateTemplate;

import java.io.IOException;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.PostConstruct;


/**
 * Ta klasa używa jBPM do zarządzania cyklem życia nowego obiektu typu {@link com.apress.springrecipes.jbpm.jbpm4.customers.Customer} 
 * w ramach przepływu pracy.
 *
 */
public class CustomerServiceImpl implements CustomerService {
    private static final String REGISTER_CUSTOMER_PROCESS_KEY = "RegisterCustomer";
    private static final Logger logger = Logger.getLogger(CustomerServiceImpl.class);
    private List<String> processDefinitions;
    @Autowired
    private ProcessEngine processEngine;
    @Value("#{ processEngine.getRepositoryService() }")
    private RepositoryService repositoryService;
    @Value("#{ processEngine.getExecutionService() }")
    private ExecutionService executionService;
    @Autowired
    private HibernateTemplate hibernateTemplate;

    @PostConstruct
    public void setupProcessDefinitions() {
        try {
            for (String processDefinition : processDefinitions) {
                NewDeployment deployment = repositoryService.createDeployment();
                deployment.addResourceFromUrl(new ClassPathResource(processDefinition).getURL());
                deployment.deploy();
            }
        } catch (IOException e) {
            logger.info("Wystąpił wyjątek IOException: " + ExceptionUtils.getFullStackTrace(e));
            throw new RuntimeException("Wystąpił błąd przy próbie dodania definicji procesu", e);
        }
    }

    public Customer createCustomer(String email, String passphrase, String firstName, String lastName) {
        Customer customer = new Customer();
        customer.setFirstName(firstName);
        customer.setLastName(lastName);
        customer.setEmail(email);
        customer.setPassphrase(passphrase);
        customer.setAuthorized(false);
        hibernateTemplate.saveOrUpdate(customer);

        Map<String, Object> vars = new HashMap<String, Object>();
        vars.put("customerId", customer.getId());

        ProcessInstance pi = executionService.startProcessInstanceByKey(REGISTER_CUSTOMER_PROCESS_KEY, vars, Long.toString(customer.getId()));

        System.out.println("pi.started " + pi.getProcessDefinitionId());

        return customer;
    }

    /**
     * Zwraca klienta na podstawie klucza głównego
     */
    public Customer getCustomerById(Long id) {
        return (Customer) hibernateTemplate.get(Customer.class, id);
    }

    /**
	 * Mogą występować też inne przypadki użycia, nieuwzględnione w tym scenariuszu (na przykład tworzenie zasobów systemowych
	 * lub przetwarzanie danych na podstawie informacji podanych przez użytkownika w trakcie tworzenia konta).
     */
    @SuppressWarnings("unchecked")
    public void authorizeCustomer(Long customerId) {
        Customer customer = getCustomerById(customerId);
        customer.setAuthorized(true);

        String processInstanceKey = Long.toString(customerId);
        String hqlToFindProcessInstanceByProcessInstanceKey = "SELECT processInstance FROM ExecutionImpl AS processInstance WHERE  processInstance.key = :processInstanceKey";
        List<Execution> executions = hibernateTemplate.findByNamedParam(hqlToFindProcessInstanceByProcessInstanceKey, new String[] { "processInstanceKey" }, new Object[] { processInstanceKey });
        System.out.println("executions: " + executions);

        for (Execution execution : executions) {
            Execution subExecution = execution.findActiveExecutionIn("confirm-receipt-of-verification-email");
            executionService.signalExecutionById(subExecution.getId());
        }

        hibernateTemplate.saveOrUpdate(customer);
    }

    /**
	 * Mogą też istnieć inne przypadki użycia, nieobsługiwane w tej metodzie, dla których warto rozdzielić przepływ pracy.
	 * Dotyczy to na przykład wycofania dostępu specjalnych kont (FTP, e-mail itd.), współużytkowanych plików itd.
     */
    public void deauthorizeCustomer(Long customerId) {
        Customer customer = getCustomerById(customerId);
        customer.setAuthorized(false);
        hibernateTemplate.saveOrUpdate(customer);
    }

    public void sendCustomerVerificationEmail(Long customerId) {
        System.out.println("Wysyłanie e-maila weryfikacyjnego do użytkownika " + customerId);
    }

    /**
     * Prosty setter ułatwiający konfigurowanie
     *
     * @param pd
     */
    public void setProcessDefinitions(List<String> pd) {
        this.processDefinitions = pd;
    }

    public void sendWelcomeEmail(Long customerId) {
        System.out.println("Wysyłanie e-maila powitalnego do użytkownika " + customerId);
    }
}
