// Wersja końcowa
package com.apress.springrecipes.court.web;

import java.util.List;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.SessionAttributes;
import org.springframework.web.bind.support.SessionStatus;

import com.apress.springrecipes.court.domain.Player;
import com.apress.springrecipes.court.domain.Reservation;
import com.apress.springrecipes.court.domain.ReservationValidator;
import com.apress.springrecipes.court.domain.SportType;
import com.apress.springrecipes.court.service.ReservationService;


@Controller
// Wiązanie kontrolera z adresem URL /reservationForm
// Początkowy widok jest określany na podstawie nazwy zwracanej przez domyślną metodę obsługi żądań GET.
@RequestMapping("/reservationForm")
// Dodawanie do sesji obiektu typu Reservation, ponieważ jest tworzony przy uruchamianiu strony i używany po przesłaniu danych
@SessionAttributes("reservation")
public class ReservationFormController {

    private ReservationService reservationService;
    private ReservationValidator validator;

    // Podłączanie usługi i walidatora w konstruktorze
    @Autowired
    public ReservationFormController(ReservationService reservationService, 
	    ReservationValidator validator) {
        this.reservationService = reservationService;
        this.validator = validator;
    }


    // Tworzenie atrybutu modelu. Ten atrybut będzie reprezentowany jako
    // z typami sportów w formularzu reservationForm
    @ModelAttribute("sportTypes")
    public List<SportType> populateSportTypes() {
        return reservationService.getAllSportTypes();
    }

    // Kontroler zawsze szuka domyślnej metody obsługi żądań GET (jej nazwa nie ma znaczenia).
    // Tu ta metoda nosi nazwę setupForm, co ułatwia jej identyfikację
    @RequestMapping(method = RequestMethod.GET)
    public String setupForm(
            @RequestParam(required = false, value = "username") String username,
            Model model) {
	// Tworzenie obiektu typu Reservation
        Reservation reservation = new Reservation();
	    // Dodawanie gracza do rezerwacji, ponieważ potrzebne będzie pole Player
        // WAŻne -  Ten obiekt Player jest zapisywany w obiekcie Reservation w sesji.
        //          Gdyby obiekt Reservation nie działał w ramach sesji, obiekt Player trzeba
	    //          byłoby odtworzyć po przesłaniu formularza
        reservation.setPlayer(new Player(username, null));
	// Dodawanie rezerwacji do modelu, co pozwala wyświetlić ją w widoku
        model.addAttribute("reservation", reservation);
	// Zwracanie łańcucha znaków z widokiem
	// Na podstawie konfiguracji widok reservationForm jest wiązany
	// z plikiem JSP  /WEB-INF/jsp/reservationForm.jsp 
	// UWAGA: Gdyby metoda zwracała void, domyślnie szukałaby tego samego
        //       widoku reservationForm, ponieważ domyślny adres URL dla tego
        //       kontrolera to @RequestMapping("/reservationForm")
        return "reservationForm";
    }

	// Kontroler zawsze szuka domyślnej metody obsługi żądań POST (nazwa nie ma tu znaczenia),
    // gdy zgłaszane jest żądanie danego adresu URL (@RequestMapping(/reservationForm)).
    // Tu ta metoda nosi nazwę submitForm, co ułatwia jej identyfikację
    @RequestMapping(method = RequestMethod.POST)
    // Przyjmuje obiekt Reservation modelu oraz obiekty BindingResult i SessionStatus 
    public String submitForm(
            @ModelAttribute("reservation") Reservation reservation,
            BindingResult result, SessionStatus status) {
	// Zakończono sprawdzanie poprawności obiektu Reservation
        validator.validate(reservation, result);	
        if (result.hasErrors()) {
	    // Wystąpiły błędy; należy wrócić do widoku reservationForm
            return "reservationForm";
        } else {
	    // Błędów nie wykryto. Należy dodać rezerwację 
            reservationService.make(reservation);
		// Wywołanie setComplete() oznacza zakończenie przetwarzania sesji.
		// Można usunąć atrybuty sesji.
	    status.setComplete();
	    // Przekierowanie do adresu URL reservationSuccess zdefiniowanego w ReservationSuccessController
            return "redirect:reservationSuccess";
        }
    }
}
