/*
  Przykład zapisu SonMicro RFID 
  Kontekst: Processing
*/


// import bibliotek:
import processing.serial.*;
import sonMicroReader.*;

String tagID = "";         // ciąg dla ID znacznika 
Serial myPort;             // instancja portu szeregowego 
SonMicroReader myReader;   // instancja SonMicroReader 

int lastCommand = 0;       // ostatnie wysłane polecenie 
int lastTagType = 0;       // typ ostatniego otrzymanego znacznika 
int lastPacketLength = 0;  // długość ostatniego otrzymanego pakietu
String lastTag = null;     // ID ostatniego otrzymanego znacznika
int lastErrorCode = 0;     // kod ostatniego otrzymanego błędu
int[] lastResponse = null; // ostatnia odpowiedź z czytnika 
                           // (nieprzetworzone dane)
int lastAntennaPower = 0;  // ostatnie otrzymane zasilanie anteny
int lastChecksum = 0;      // ostatnia otrzymana suma kontrolna 

int fontHeight = 14;       // wysokość czcionki dla tekstu na ekranie 
String message = null;     // wiadomość odczytana ze znacznika
String outputString = "Witaj, świecie!"; // ciąg do zapisania 
                                         // do znacznika 

// motyw kolorów: Ghostly Music 
// autor: banshee prime, http://kuler.adobe.com
color currentcolor = #CBD0D4;        // bieżący kolor przycisku
color highlight = #745370;
color buttoncolor = #968195;
color userText = #444929;
color buttonText = #ACB0B9;

ArrayList buttons = new ArrayList(); // lista przycisków 
// same przyciski:
String[] buttonNames = {
  "zasilanie anteny", "wybierz znacznik", "uwierzytelnianie", "odczyt bloku", "szukaj znacznika", 
  "zapisz blok", "wersja firmware"
};

void setup() {
  // ustaw rozmiar okna:
  size(600, 400);
  // lista wszystkich portów szeregowych:
  println(Serial.list());

  // w oparciu o listę portów szeregowych 
  // wyświetloną przez poprzednie polecenie, 
  // zmień 0 na numer Twojego portu:
  String portnum = Serial.list()[0];
  // zainicjuj port szeregowy; domyślna 
  // szybkość danych dla czytnika SM130 to 19200:
  myPort = new Serial(this, portnum, 19200);
  // zainicjuj instancję czytnika:
  myReader = new SonMicroReader(this, myPort);
  myReader.start();

  // utwórz czcionkę z drugiej czcionki dostępnej w systemie:
  PFont myFont = createFont(PFont.list()[2], fontHeight);
  textFont(myFont);
  // utwórz przyciski poleceń:
  makeButtons();
}

void draw() {
  background(currentcolor);
  // narysuj przyciski poleceń:
  drawButtons();
  // narysuj pola danych wyjściowych:
  textAlign(LEFT);
  text("Polecenie: " + hex(lastCommand, 2), 10, 30);
  text("Długość pakietu: " +lastPacketLength, 10, 50);
  text("Zasilanie anteny: " + lastAntennaPower, 10, 70);
  text("Typ znacznika: " + lastTagType, 10, 90);
  text("Ciąg znacznika: " + lastTag, 10, 110);
  text("Kod błędu: " + hex(lastErrorCode, 2), 10, 130);

  // wyświetl wartości szesnastkowe dla wszystkich 
  // bajtów w odpowiedzi:
  String responseString = "";
  if (lastResponse != null) {
    for (int b = 0; b < lastResponse.length; b++) {
      responseString += hex(lastResponse[b], 2);
      responseString += " ";
    }
    // zawijaj pełny tekst tak, by nie nadpisać 
    // przycisków i nie zaśmiecać ekranu:
    text("Pełna odpowiedź:\n" + responseString, 10, 150, 300, 200);
  }
  // wyświetl komunikaty o błędach z czytnika:
  text(myReader.getErrorMessage(), 10, 210);
  // wyświetl ostatnią wiadomość odczytaną ze znacznika:
  text("Ostatnia wiadomość odczytana ze znacznika:\n" + message, 10, 230);

  // wyświetl wiadomość wyjściową:
  text("Wpisz wiadomość do zapisu do znacznika:\n", 10, 300);
  fill(userText);
  text(outputString, 10, 320);

  // pokaż wersję biblioteki:
  fill(0);
  text("Wersja SonMicroReader: " + myReader.version(),
  width - 300, height - 30);
}

/*
  Ta funkcja jest wywoływana automatycznie, kiedy tylko 
  pojawia się prawidłowy pakiet danych z czytnika 
*/
void sonMicroEvent(SonMicroReader myReader) {
  // uzyskaj wszystkie ważne dane z ostatniego pakietu danych:
  lastCommand = myReader.getCommand();
  lastTagType = myReader.getTagType();
  lastPacketLength = myReader.getPacketLength();
  lastTag = myReader.getTagString();
  lastErrorCode = myReader.getErrorCode();
  lastAntennaPower = myReader.getAntennaPower();
  lastResponse = myReader.getSonMicroReading();
  lastChecksum = myReader.getCheckSum();

  // jeśli ostatnim wysłanym poleceniem było 
  // polecenie odczytu bloku:
  if (lastCommand == 0x86) {
    int[] inputString = myReader.getPayload();
    message = "";
    for (int c = 0; c < inputString.length; c++) {
      message += char(inputString[c]);
    }
  }
}

/*
  Jeśli klawisz zostanie naciśnięty, dodaj go do wyjściowego 
  ciągu znaków lub usuń ciąg znaków, jeśli jest to backspace:
*/
void keyTyped() {
  switch (key) {
  case BACKSPACE:         // usuń
    outputString = "\0";
    break;
  default:
    if (outputString.length() < 16) {
      outputString += key;
    }
    else {
      outputString = "Ciąg wyjściowy nie może być dłuższy niż 16 znaków";
    }
  }
}

/*
  zainicjuj wszystkie przyciski 
*/
void makeButtons() {
  // zdefiniuj i utwórz prostokątny przycisk 
  for (int b = 0; b < buttonNames.length; b++) {
    // utwórz nowy przycisk o nazwie następnego na liście:
    Button thisButton = new Button(400, 30 +b*30,
                                   150, 20,
                                   buttoncolor, highlight, buttonNames[b]);
    buttons.add(thisButton);
  }
}

/*
  Narysuj wszystkie przyciski. 
*/
void drawButtons() {
  for (int b = 0; b < buttons.size(); b++) {
    // pobierz przycisk z Arraylist:
    Button thisButton = (Button)buttons.get(b);
    // aktualizuj jego status przyciśnięcia:
    thisButton.update();
    // narysuj przycisk:
    thisButton.display();
}
}

void mousePressed() {
  // iteruj po przyciskach, 
  // uaktywnij ten, który jest naciśnięty
  for (int b = 0; b < buttons.size(); b++) {
    Button thisButton = (Button)buttons.get(b);
    if (thisButton.containsMouse()) {
      doButtonAction(thisButton);
    }
  }
}

/*
  Jeśli któryś z przycisków poleceń jest wciśnięty, sprawdź, 
  który to, i podejmij odpowiednie działania.
*/
void doButtonAction(Button thisButton) {
  // sprawdź, który to przycisk w ArrayList:
  int buttonNumber = buttons.indexOf(thisButton);

  // zrób, co trzeba:
  switch (buttonNumber) {
  case 0:          // ustaw zasilanie anteny
    if (myReader.getAntennaPower() < 1) {
      myReader.setAntennaPower(0x01);
    }
    else {
      myReader.setAntennaPower(0x00);
    }
    break;
  case 1:        // wybierz znacznik
    myReader.selectTag();
    break;
  case 2:       // uwierzytelnianie 
    myReader.authenticate(0x04, 0xFF);
    break;
  case 3:       // odczyt bloku 
    myReader.readBlock(0x04);
    break;
  case 4:      // szukanie znacznika 
    myReader.seekTag();
    break;
  case 5:      // zapisz znacznik - musi być 16 bajtów lub mniej 
    myReader.writeBlock(0x04, outputString);
    outputString = "";
    break;
  case 6:      // uzyskaj wersję oprogramowania układowego czytnika
    myReader.getFirmwareVersion();
    break;
}
}

class Button {
  int x, y, w, h;                  // pozycje przycisków 
  color basecolor, highlightcolor; // kolor i kolor podświetlenia 
  color currentcolor;              // bieżący kolor przycisku 
  String name;

  // konstruktor: ustawia wszystkie wartości 
  // początkowe dla każdej instancji klasy Button 
  Button(int thisX, int thisY, int thisW, int thisH,
      color thisColor, color thisHighlight, String thisName) {
    x = thisX;
    y = thisY;
    h = thisH;
    w = thisW;
    basecolor = thisColor;
    highlightcolor = thisHighlight;
    currentcolor = basecolor;
    name = thisName;
  }

  // jeśli kursor znajduje się nad przyciskiem, zmień kolor przycisku:
  void update() {
    if (containsMouse()) {
      currentcolor = highlightcolor;
    }
    else {
      currentcolor = basecolor;
    }
  }

  // rysuj przycisk i jego tekst:
  void display() {
    fill(currentcolor);
    rect(x, y, w, h);
    // umieść nazwę na środku przycisku:
    fill(0);
    textAlign(CENTER, CENTER);
    text(name, x+w/2, y+h/2);
  }

  // sprawdź, czy pozycja myszy jest wewnątrz 
  // granic prostokąta:
  boolean containsMouse() {
    if (mouseX >= x && mouseX <= x+w &&
      mouseY >= y && mouseY <= y+h) {
      return true;
    }
    else {
      return false;
    }
  }
}
