/** Definicja cechy, która pozwala zaznaczyć, 
 * że klasa pozwala na podłączanie zewnętrznych obserwatorów, 
 * którzy będą powiadamiani o zmianach.
 */
trait Observable {
  // Definicja typu zwracanego po zarejestrowaniu wywołania zwrotnego.
  type Handle <: {
    def remove() : Unit
  }
  /** Wywołania zwrotne zarejestrowane dla tej instancji Observable */
  protected var callbacks = Map[Handle, this.type => Unit]()

   /** Rejestracja nowego obserwatora klasy. 
   * Obserwatorzy to proste funkcje, które pobierają tę klasę i nic nie zwracają. 
   * W wyniku rejestracji zwracany jest uchwyt, 
   * który obserwatorowi na wypisanie się z tej instancji Observable.
   */
  def observe(callback : this.type => Unit) : Handle = {
    val handle = createHandle(callback)
    callbacks += (handle -> callback)
    handle
  }
  /** Usuwa obserwatora. */
  def unobserve(handle : Handle) : Unit = {
    callbacks -= handle
  }
  /** Ta metoda jest wywoływana przez podklasy po zmianie stanu,
   * co umożliwia powiadomienie obserwatorów.
   */
  protected def notifyListeners() : Unit =
    for(callback <- callbacks.values) callback(this)

  /**
   * Podklasy przesłaniają tę metodę w celu zdefiniowania własnego schematu ujednoznaczniania wywołań zwrotnych.
   */
  protected def createHandle(callback : this.type => Unit) : Handle
}

/** Ta cecha definiuje domyślną implementację typu uchwytu. */
trait DefaultHandles extends Observable {
  /** Prosta implementacja uchwytu.  */
  class HandleClass {
    def remove() {
      DefaultHandles.this.unobserve(this)
    }
  }
  type Handle = HandleClass
  /** Każde wywołanie zwrotne otrzymuje nowy uchwyt. */
  protected def createHandle(callback : this.type => Unit) : Handle = new HandleClass
}

/** Prosta definicja magazynu (cache) przechowującego pojedynczą wartość. 
 * Magazyn jest obserwowalny; obserwatorzy zostaną powiadomieni, gdy zmieni się przechowywana wartość.
 */
class VariableStore[X](private var value : X) extends Observable with DefaultHandles {
  /** Pobierz przechowywaną wartość */
  def get : X = value
  /** Zmień wartość. Doprowadzi to do powiadomienia obserwatorów. */
  def set(newValue : X) : Unit = {
    value = newValue
    notifyListeners()
  }
  // Przesłonięta na potrzeby REPL.
  override def toString : String = "VariableStore(" + value + ")"
}

 /** Ta cecha definiuje mechanizm obsługi uchwytów pozwalający na 
 * wypisanie obserwatorów, gdy aktualna klasa 'zakończy działanie'.
 */
trait Dependencies {
  // Ten typ pozwala na odwołanie się do każdego uchwytu z każdej instancji Observable.   
  // Ponieważ uchwyt jest definiowany poza Observable,
  // stosujemy odwołanie egzystencjalne do Observable. 
  type Ref = x.Handle forSome { val x: Observable }
  /** Obecnie zarejestrowani obserwatorzy. */
  private var handles = List[Ref]()
  /** Dodanie nowego uchwytu */
  protected def addHandle(handle : Ref) : Unit = {
    handles :+= handle
  }
  /** Usunięcie wszystkich obserwatorów w oparciu o uchwyty*/
  protected def removeDependencies() {
    for(h <- handles) h.remove()
    handles = List()
  }
   /** Ta metoda działa jak Observable.observe, ale
   * rejestruje obserwatora *i* dodaje go do listy zależności. 
   */
  protected def observe[T <: Observable](obj : T)(handler : T => Unit) : Ref = {
    val ref = obj.observe(handler)
    addHandle(ref)
    ref
  }
}


/*

scala> val x = new VariableStore(12)
x: VariableStore[Int] = VariableStore(12)

scala> val d = new Dependencies {}
d: java.lang.Object with Dependencies = $anon$1@153e6f83


Uwaga: Przez błąd w Scali 2.8.x, poniższy kod nie zadziała.

scala> d.addHandle(x.observe(println))
<console>:8: error: type mismatch;
 found   : x.Handle
 required: d.Ref
       d.addHandle(x.observe(println))
                            ^
// Poniższe wywołania zadziałają

scala> val t = x.observe(println)
t: x.Handle = DefaultHandles$HandleClass@662fe032

scala> d.addHandle(t)

scala> val t2 = x.observe(println)
t2: x.Handle = DefaultHandles$HandleClass@57530551

scala> d.addHandle(t2)

scala> x.set(1)
VariableStore(1)
VariableStore(1)

scala> d.removeDependencies()

scala> x.set(2)

*/
