package scalax.functional


/**
 * Ta klasa reprezentuje funktor z teorii kategorii. 
 * Potrafi przekształcać elementy z kategorii wszystkich typów do podkategorii typów T[_].
 */
trait Functor[T[_]] {
   /** Ta metoda mapuje daną wartość do kategorii funktora. Odpowiada konstruktorowi.*/    
   def apply[A](x: A): T[A]
   /** Ta metoda pobiera niezmapowany morfizm i przekłada go na nową dziedzinę.
    * @param x Mapowana wartość
    * @param f Morfizm w oryginalnej dziedzinie
    * @return Nowa wartość w wynikowej dziedzinie
    */
   def map[A,B](x: T[A])(f: A=>B): T[B]
}


object Functor {
  // Wartości domyślne dla biblioteki standardowej. 
  implicit object TraversableFunctor extends Functor[Traversable] {
    override def apply[A](x: A) = Traversable(x)
    override def map[A,B](x:Traversable[A])(f: A=>B) = x map f
  }
  implicit object OptionFunctor extends Functor[Option] {
    override def apply[A](x: A) = Some(x)
    override def map[A,B](x: Option[A])(f: A=>B) = x map f
  }
}

/**
 * Ta klasa przechowuje 'podrasowane' funkcje stosowane na wartościach F[A] jeśli F jest funktorem.
 */
final class FunctorOps[F[_], A](val value: F[A], val functor: Functor[F]) {
  @inline final def map[B](f: A=>B): F[B] = functor.map(value)(f)
}

/**
 * Ta cecha przechowuje wszystkie niejawne widoki, zapewniając wygodną składnię dla funktorów.
 */
trait FunctorImplicits {
  implicit def funcOperations[F[_], A](value : F[A])(implicit functor: Functor[F]): FunctorOps[F,A] =
    new FunctorOps[F,A](value, functor)
}

