import os
import requests
import re
import pandas as pd
import yaml

from bs4 import BeautifulSoup


URL = 'https://docs.microsoft.com/en-us/power-bi/connect-data/service-python-packages-support'
page = requests.get(URL)    # wykonuje żądanie HTTP do danego adresu URL

soup = BeautifulSoup(page.content, 'html.parser')   # pobiera sparsowany obiekt HTML

main_soup = soup.find(id='main') # znajdź znacznik z id = 'main'

#-------------------------------------------------------------------
# Pobierz właściwą wersję Pythona dla Power BI Service Python Visuals
#-------------------------------------------------------------------

# Znajdź znacznik <li> zawierający 'Python runtime' w tekście, a następnie usuń wiodące i końcowe spacje za pomocą funkcji strip()
python_ver_full_str = main_soup.find('li', text=re.compile('Python runtime')).text.strip()

# Za pomocą regexa wyodrębnij ciąg wersji         
m = re.search('(?<=Python\s)\d{1,3}\.\d{1,3}(\.\d{1,3})?', python_ver_full_str)

python_ver_str = m.group(0) # pobierz wyodrębniony tekst z grupy domyślnej (0)


#---------------------------------------------------------------------
# Pobierz tabelę zawierającą wszystkie pakiety Pythona i ich wersje
#---------------------------------------------------------------------

# Celem jest bezpieczne znalezienie odpowiedniej komórki tabeli (pakiet pandas z pewnością tam będzie).
# Z tej komórki możesz następnie wrócić do obiektów nadrzędnych, dopóki nie odwołasz się do całej tabeli.

# Znajdź wszystkie tagi <td> zawierające w tekście słowo 'pandas', a następnie pobierz pierwszy z nich
pandas_cell_soup = main_soup.findAll('td', text=re.compile('pandas'))[0]

# Zacznij od tej komórki i wróć do tabeli nadrzędnej
packages_table_soup = pandas_cell_soup.parent.parent.parent

# Pobierz wszystkie elementy wierszy tabeli, w tym nagłówki (ciało)
packages_body_soup = packages_table_soup.find_all('tr')

# Wyodrębnij z ciała pierwszy element wiersza: to nagłówek
packages_header_soup = packages_body_soup[0]

# Wyodrębnij z ciała wszystkie elementy wierszy z wyjątkiem pierwszego: mamy wiersze tabeli
packages_rows_soup = packages_body_soup[1:]

# Parsujemy nagłówek, aby pobrać nazwy kolumn tabeli na liście
column_names = []
for item in packages_header_soup.find_all('th'):# przetwarzaj w pętli wszystkie elementy
    item = (item.text).rstrip('\n')             # konwertuj elementy na tekst i obetnij "\n"
    column_names.append(item)                   # dołącz do listy column_names oczyszczoną nazwę kolumny
    
# Parsowanie wszystkich elementów wierszy, aby pobrać do listy wiersze tabeli
packages_rows = []
for row_idx in range(len(packages_rows_soup)):  # przetwarzaj w pętli wszystkie elementy wierszy, korzystając z ich indeksu 
    row = []                                    # ta lista będzie zawierać komórki danych dla jednego wiersza 
    for row_item_soup in packages_rows_soup[row_idx].find_all('td'): # przetwarzaj w pętli wszystkie komórki danych ustalonego wiersza
        
        # Usuń z danych komórki znaki \xa0 (spacje niełamiące), \n (znaki nowego wiersza), \\s dowolne znaki białych spacji
        # (pakiety nie mają spacji w nazwie) oraz przecinki (separatory tysięcy) z row_item.text
        # (oczyszczone dane komórki) z wykorzystaniem regexa
        cell_text = re.sub('(\xa0)|(\n)|,|\\s', '', row_item_soup.text)
        row.append(cell_text) # dołącz cell_text to listy wiersza
        
    packages_rows.append(row) # dołącz jeen wiersz do packages_rows

# Użyj sparsowanych list wierszy i nazw kolumn, aby utworzyć strukturę DataFrame pakietu pandas
df = pd.DataFrame(data=packages_rows,columns=column_names)


#----------------------------------------------------
# Za pomocą struktury DataFrame napisz plik YAML dla środowiska
#----------------------------------------------------

# Usuń nieużywaną kolumnę Link
packages_df = df.drop('Link', axis=1)

packages_version_lst = packages_df.agg('=='.join, axis=1).values.tolist()

# Zapisz plik YAML z wykorzystaniem pobranych informacji o pakietach
packages_dict = {
    'name': 'pbi_visuals_env',

    'dependencies': [
        'python==%s' % python_ver_str,
        'pip',
        {'pip': packages_version_lst}
    ]
    
}

print( yaml.dump(packages_dict, default_flow_style=False) )

destination_path = r'.'

## Jeśli chcesz utworzyć podfolder 
#os.makedirs(destination_path, exist_ok=True)

yaml_file_name = 'visuals_environment.yaml'

with open(os.path.join(destination_path, yaml_file_name), 'w') as file:
    documents = yaml.dump(packages_dict, file)


