#!/usr/bin/env python
# coding: utf-8

# 
# Kody źródłowe do książki: Python. Uczenie maszynowe w przykładach
#  
# Rozdział 5.: Prognozowanie cen akcji za pomocą algorytmów regresji
#  
# Autor: Yuxi (Hayden) Liu (yuxi.liu.ece@gmail.com)
# 

# # Pozyskiwanie cen akcji

# ## Pozyskiwanie danych i generowanie cech

import pandas as pd


mydata = pd.read_csv('20051201_20051210.csv', index_col='Date')
mydata


def add_original_feature(df, df_new):
    df_new['open'] = df['Open']
    df_new['open_1'] = df['Open'].shift(1)
    df_new['close_1'] = df['Close'].shift(1)
    df_new['high_1'] = df['High'].shift(1)
    df_new['low_1'] = df['Low'].shift(1)
    df_new['volume_1'] = df['Volume'].shift(1)
    


def add_avg_price(df, df_new): 
    df_new['avg_price_5'] = df['Close'].rolling(5).mean().shift(1)
    df_new['avg_price_30'] = df['Close'].rolling(21).mean().shift(1)
    df_new['avg_price_365'] = df['Close'].rolling(252).mean().shift(1)
    df_new['ratio_avg_price_5_30'] = df_new['avg_price_5'] / df_new['avg_price_30']
    df_new['ratio_avg_price_5_365'] = df_new['avg_price_5'] / df_new['avg_price_365']
    df_new['ratio_avg_price_30_365'] = df_new['avg_price_30'] / df_new['avg_price_365']
    


def add_avg_volume(df, df_new):
    df_new['avg_volume_5'] = df['Volume'].rolling(5).mean().shift(1)
    df_new['avg_volume_30'] = df['Volume'].rolling(21).mean().shift(1)
    df_new['avg_volume_365'] = df['Volume'].rolling(252).mean().shift(1)
    df_new['ratio_avg_volume_5_30'] = df_new['avg_volume_5'] / df_new['avg_volume_30']
    df_new['ratio_avg_volume_5_365'] = df_new['avg_volume_5'] / df_new['avg_volume_365']
    df_new['ratio_avg_volume_30_365'] = df_new['avg_volume_30'] / df_new['avg_volume_365']
    


def add_std_price(df, df_new):
    df_new['std_price_5'] = df['Close'].rolling(5).std().shift(1)
    df_new['std_price_30'] = df['Close'].rolling(21).std().shift(1)
    df_new['std_price_365'] = df['Close'].rolling(252).std().shift(1)
    df_new['ratio_std_price_5_30'] = df_new['std_price_5'] / df_new['std_price_30']
    df_new['ratio_std_price_5_365'] = df_new['std_price_5'] / df_new['std_price_365']
    df_new['ratio_std_price_30_365'] = df_new['std_price_30'] / df_new['std_price_365']
    


def add_std_volume(df, df_new):
    df_new['std_volume_5'] = df['Volume'].rolling(5).std().shift(1)
    df_new['std_volume_30'] = df['Volume'].rolling(21).std().shift(1)
    df_new['std_volume_365'] = df['Volume'].rolling(252).std().shift(1)
    df_new['ratio_std_volume_5_30'] = df_new['std_volume_5'] / df_new['std_volume_30']
    df_new['ratio_std_volume_5_365'] = df_new['std_volume_5'] / df_new['std_volume_365']
    df_new['ratio_std_volume_30_365'] = df_new['std_volume_30'] / df_new['std_volume_365']
    


def add_return_feature(df, df_new):
    df_new['return_1'] = ((df['Close'] - df['Close'].shift(1)) / df['Close'].shift(1)).shift(1)
    df_new['return_5'] = ((df['Close'] - df['Close'].shift(5)) / df['Close'].shift(5)).shift(1)
    df_new['return_30'] = ((df['Close'] - df['Close'].shift(21)) / df['Close'].shift(21)).shift(1)
    df_new['return_365'] = ((df['Close'] - df['Close'].shift(252)) / df['Close'].shift(252)).shift(1)
    df_new['moving_avg_5'] = df_new['return_1'].rolling(5).mean().shift(1)
    df_new['moving_avg_30'] = df_new['return_1'].rolling(21).mean().shift(1)
    df_new['moving_avg_365'] = df_new['return_1'].rolling(252).mean().shift(1)
    


def generate_features(df):
    """
    Funkcja generująca cechy na podstawie historycznych
    wartości indeksu i jego zmienności
    @param df: obiekt DataFrame zawierający kolumny
        "Open", "Close", "High", "Low", "Volume", "Adjusted Close"
    @return: obiekt DataFrame zawierający zbiór danych z nowymi cechami
    """
    df_new = pd.DataFrame()
    # 6 oryginalnych cech
    add_original_feature(df, df_new)
    # 31 wygenerowanych cech
    # średnia cena
    add_avg_price(df, df_new)
    # średni wolumen
    add_avg_volume(df, df_new)
    # odchylenie standardowe ceny
    add_std_price(df, df_new)
    # odchylenie standardowe wolumenu
    add_std_volume(df, df_new)
    # # wartość wynikowa
    add_return_feature(df, df_new)
    # wartości docelowe
    df_new['close'] = df['Close']
    df_new = df_new.dropna(axis=0)
    return df_new


data_raw = pd.read_csv('19900101_20230630.csv', index_col='Date')
data = generate_features(data_raw)
print(data.round(decimals=3).head(5))


# # Szacowanie za pomocą regresji liniowej

# ## Implementacja regresji liniowej od podstaw

import numpy as np


def compute_prediction(X, weights):
    """
    Funkcja wyliczająca prognozę y_hat na podstawie bieżących wag
    """
    return np.dot(X, weights)


def update_weights_gd(X_train, y_train, weights, learning_rate):
    """
    Funkcja aktualizuje wagi o jeden krok i zwraca nowe wartości
    """
    predictions = compute_prediction(X_train, weights)
    weights_delta = np.dot(X_train.T, y_train - predictions)
    m = y_train.shape[0]
    weights += learning_rate / float(m) * weights_delta
    return weights


def compute_loss(X, y, weights):
    """
    Funkcja wyliczająca koszt J(w) 
    """
    predictions = compute_prediction(X, weights)
    return np.mean((predictions - y) ** 2 / 2.0)


def train_linear_regression(X_train, y_train, max_iter, learning_rate, fit_intercept=False, display_loss=500):
    """
    Funkcja trenująca model regresji liniowej z wykorzystaniem
    gradientu prostego i zwracająca wytrenowany model
    """
    if fit_intercept:
        intercept = np.ones((X_train.shape[0], 1))
        X_train = np.hstack((intercept, X_train))
    weights = np.zeros(X_train.shape[1])
    for iteration in range(max_iter):
        weights = update_weights_gd(X_train, y_train, weights, learning_rate)
        # Wyświetlenie kosztu co określoną liczbe iteracji (500 domyślnie)
        if iteration % display_loss == 0:
            print(compute_loss(X_train, y_train, weights))
    return weights


def predict(X, weights):
    if X.shape[1] == weights.shape[0] - 1:
        intercept = np.ones((X.shape[0], 1))
        X = np.hstack((intercept, X))
    return compute_prediction(X, weights)


# Prosty przykład
X_train = np.array([[6], [2], [3], [4], [1], [5], [2], [6], [4], [7]])
y_train = np.array([5.5, 1.6, 2.2, 3.7, 0.8, 5.2, 1.5, 5.3, 4.4, 6.8])


weights = train_linear_regression(X_train, y_train, max_iter=100, learning_rate=0.01, fit_intercept=True)


X_test = np.array([[1.3], [3.5], [5.2], [2.8]])

predictions = predict(X_test, weights)

import matplotlib.pyplot as plt
plt.scatter(X_train[:, 0], y_train, marker='o', c='b')
plt.scatter(X_test[:, 0], predictions, marker='*', c='k')
plt.xlabel('x')
plt.ylabel('y')
plt.show()


# Przykład z diagnozowaniem cukrzycy
from sklearn import datasets
diabetes = datasets.load_diabetes()
print(diabetes.data.shape)

num_test = 30
X_train = diabetes.data[:-num_test, :]
y_train = diabetes.target[:-num_test]


weights = train_linear_regression(X_train, y_train, max_iter=5000, learning_rate=1, fit_intercept=True)

X_test = diabetes.data[-num_test:, :]
y_test = diabetes.target[-num_test:]

predictions = predict(X_test, weights)

print(predictions)
print(y_test)


# ## Implementacja regresji liniowej z wykorzystaniem pakietu scikit-learn

# Bezpośrednie użycie klasy SGDRegressor ze scikit-learn
from sklearn.linear_model import SGDRegressor
regressor = SGDRegressor(loss='squared_error', 
                         penalty='l2', 
                         alpha=0.0001, 
                         learning_rate='constant', 
                         eta0=0.2,
                         max_iter=100, 
                         random_state=42)


regressor.fit(X_train, y_train)
predictions = regressor.predict(X_test)
print(predictions)


# ## Implementacja regresji liniowej z wykorzystaniem pakietu TensorFlow

import tensorflow as tf


layer0 = tf.keras.layers.Dense(units=1, input_shape=[X_train.shape[1]])
model = tf.keras.Sequential(layer0)


model.compile(loss='mean_squared_error',
              optimizer=tf.keras.optimizers.Adam(1))


model.fit(X_train, y_train, epochs=100, verbose=True)


predictions = model.predict(X_test)[:, 0]
print(predictions)


# ---

# Czytelnicy mogą pominąć następną komórkę.

get_ipython().system('jupyter nbconvert --to python ch5_part1.ipynb --TemplateExporter.exclude_input_prompt=True')

