import json
import statistics

import pyrg_funkcje


FOLDER = "daneztreningow/"
CHARTS = "wykresy/"
JSONDATA = "rekordy.json"


def event_lookup(event):
    """Wykonuje konwersję nazw plików na klucze słownika rekordów świata.
    
    Na podstawie podanej nazwy konkurencji (nazwy pliku z wynikami pływaka)
    funkcja konwertuje ją do postaci łańcucha, który może zostać użyty do 
    pobrania danych ze słownika "records"
    """
    conversions = {
        "kraul": "dowolnym",
        "grzbietowy": "grzbietowym",
        "klasyczny": "klasycznym",
        "motyl": "motylkowym",
        "zmienny": "zmiennym",
    }

    *_, distance, stroke = event.removesuffix(".txt").split("-")
    return f"{distance[:-1]} m stylem {conversions[stroke]}"


def read_swim_data(filename):
    """Zwraca czasy z pliku.

    Na podstawie nazwy pliku pływaka, funkcja pobiera dane ze wskazanego pliku
    i zwraca je w formie krotki
    """
    swimmer, age, distance, stroke = filename.removesuffix(".txt").split("-")
    with open(FOLDER + filename) as file:
        lines = file.readlines()
        times = lines[0].strip().split(";")
    converts = []
    for t in times:
        # Może brakować wartości minut, musimy się przed tym zabezpieczyć
        if ":" in t:
            minutes, rest = t.split(":")
            seconds, hundredths = rest.split(",")
        else:
            minutes = 0
            seconds, hundredths = t.split(",")
        converts.append(
            (int(minutes) * 60 * 100) + (int(seconds) * 100) + int(hundredths)
        )
    average = statistics.mean(converts)
    mins_secs, hundredths = f"{(average / 100):.2f}".split(".")  
    mins_secs = int(mins_secs)
    minutes = mins_secs // 60
    seconds = mins_secs - minutes * 60
    average = f"{minutes}:{seconds:0>2},{hundredths}"  # Twój pierwszy f-łańcuch

    return swimmer, age, distance, stroke, times, average, converts  # Zwracamy krotkę



def produce_bar_chart(fn, location=CHARTS):
    """Na podstawie nazwy pliku pływaka, generuje wykres słupkowy, używając HTML i SVG.

    Wykres jest zapisywany w katalogu określonym przez stałą CHATRS. Funkcja zwraca
    ścieżkę dostępu do pliku wykresu.
    """
    swimmer, age, distance, stroke, times, average, converts = read_swim_data(fn)
    from_max = max(converts)
    times.reverse()
    converts.reverse()
    title = f"{swimmer} (poniżej {age} lat), {distance}, styl: {stroke}"
    header = f"""<!DOCTYPE html>
                    <html>
                        <head>
                            <title>{title}</title>
                            <link rel="stylesheet" href="/static/webapp.css"/>
                        </head>
                        <body>
                            <h2>{title}</h2>"""
    body = ""
    for n, t in enumerate(times):
        bar_width = pyrg_funkcje.convert2range(converts[n], 0, from_max, 0, 350)
        body = body + f"""
                            <svg height="30" width="400">
                                <rect height="30" width="{bar_width}" style="fill:rgb(0,0,255);" />
                            </svg>{t}<br />"""
   
    with open(JSONDATA) as jf:
        records = json.load(jf)
    COURSES = ("Mężczyźni, basen 50 m", "Kobiety, basen 50 m", 
               "Mężczyźni, basen 25 m", "Kobiety, basen 25 m")
    times = []
    for course in COURSES:
        times.append(records[course][event_lookup(fn)])
                     
    footer = f"""
                            <p>Średni czas: {average}</p>
                            <p>M: {times[0]} ({times[2]})<br />K: {times[1]} ({times[3]})</p>
                        </body>
                    </html>"""
    page = header + body + footer
    save_to = f"{location}{fn.removesuffix('.txt')}.html"
    with open(save_to, "w") as sf:
        print(page, file=sf)

    return save_to
