// python_interface.c
// - Interfejs modułu cpython dla diffusion.c

#define NPY_NO_DEPRECATED_API    NPY_1_7_API_VERSION

#include <Python.h>
#include <numpy/arrayobject.h>
#include "diffusion.h"

/* Docstrings */
static char module_docstring[] =
   "Zapewnia zoptymalizowaną metodę rozwiązywania równania dyfuzji";
static char cdiffusion_evolve_docstring[] =
   "Rozwijanie siatki dwuwymiarowej za pomocą równania dyfuzji";

PyArrayObject *py_evolve(PyObject *, PyObject *);

/* Specyfikacja modułu  */
static PyMethodDef module_methods[] =
{
   /* { method name , C function              , argument types , docstring                   } */
   { "evolve", (PyCFunction)py_evolve, METH_VARARGS, cdiffusion_evolve_docstring },
   { NULL,     NULL,                              0, NULL                        }
};

static struct PyModuleDef cdiffusionmodule =
{
   PyModuleDef_HEAD_INIT,
   "cdiffusion",      /* Nazwa modułu */
   module_docstring,  /* Dokumentacja modułu. Może być wartość NULL. */
   -1,                /* Wielkość stanu modułu przypadającego na interpreter
                       * lub wartość -1, jeśli moduł utrzymuje stan w zmiennych globalnych. */
   module_methods
};

PyArrayObject *py_evolve(PyObject *self, PyObject *args)
{
   PyArrayObject *data;
   PyArrayObject *next_grid;
   double         dt, D = 1.0;

   /* Funkcja „rozwijania” będzie mieć sygnaturę:
    *     evolve(data, next_grid, dt, D=1)
    */
   if (!PyArg_ParseTuple(args, "OOd|d", &data, &next_grid, &dt, &D))
   {
      PyErr_SetString(PyExc_RuntimeError, "Niepoprawne argumenty");
      return(NULL);
   }

   /* Sprawdzenie, czy tablice narzędzia numpy są ciągłe w pamięci */
   if (!PyArray_Check(data) || !PyArray_ISCONTIGUOUS(data))
   {
      PyErr_SetString(PyExc_RuntimeError, "Tablica data nie jest ciągłą tablicą.");
      return(NULL);
   }
   if (!PyArray_Check(next_grid) || !PyArray_ISCONTIGUOUS(next_grid))
   {
      PyErr_SetString(PyExc_RuntimeError, "Tablica next_grid nie jest ciągłą tablicą.");
      return(NULL);
   }

   /* Sprawdzenie, czy tablice siatki data i next_grid są takiego samego typu i mają identyczne wymiary.
   */
   if (PyArray_TYPE(data) != PyArray_TYPE(next_grid))
   {
      PyErr_SetString(PyExc_RuntimeError,
                      "Tablice next_grid i data powinny być tego samego typu.");
      return(NULL);
   }
   if (PyArray_NDIM(data) != 2)
   {
      PyErr_SetString(PyExc_RuntimeError, "Tablica data powinna być dwuwymiarowa.");
      return(NULL);
   }
   if (PyArray_NDIM(next_grid) != 2)
   {
      PyErr_SetString(PyExc_RuntimeError, "Tablica next_grid powinna być dwuwymiarowa.");
      return(NULL);
   }
   if ((PyArray_DIM(data, 0) != PyArray_DIM(next_grid, 0)) ||
       (PyArray_DIM(data, 1) != PyArray_DIM(next_grid, 1)))
   {
      PyErr_SetString(PyExc_RuntimeError,
                      "Tablice data i next_grid muszą mieć takie same wymiary.");
      return(NULL);
   }

   evolve(
      PyArray_DATA(data),
      PyArray_DATA(next_grid),
      D,
      dt
      );

   Py_XINCREF(next_grid);
   return(next_grid);
}

/* Inicjowanie modułu */
PyMODINIT_FUNC
PyInit_cdiffusion(void)
{
   PyObject *m;

   m = PyModule_Create(&cdiffusionmodule);
   if (m == NULL)
   {
      return(NULL);
   }

   /* Ładowanie funkcji narzędzia numpy. */
   import_array();

   return(m);
}
