﻿# locationlistener.py

"""Odbieranie tweetów spełniających określone kryteria
   i zapamiętywanie ich w postaci listy słowników."""

import tweepy
from tweetutilities import get_tweet_content

class LocationListener(tweepy.StreamListener):
    """Obsługuje strumień nadchodzących tweetów
       w celu odczytywania informacji o ich lokalizacji."""
       
    def __init__(self, api, counts_dict, tweets_list, topic, limit=10):
        """Konfigurowanie obiektu-obserwatora"""

        self.tweets_list = tweets_list
        self.counts_dict = counts_dict
        self.topic = topic
        self.TWEET_LIMIT = limit
        super().__init__(api)  # wywołanie __init__ z nadklasy

    def on_status(self, status):
        """Metoda wywoływana, gdy Twitter dostarcza nowy tweet"""
       
        # odczytanie z tweetu: nazwy ekranowej nadawcy,
        # treści oraz informacji o lokalizacji
        tweet_data = get_tweet_content(status, location=True)  

        # ignorowanie retweetów oraz tweetów, 
        # w których temat nie pojawia się w treści
        if (tweet_data['text'].startswith('RT') or
            self.topic.lower() not in tweet_data['text'].lower()):
            return

        # zliczaj również tweety bez informacji o lokalizacji
        self.counts_dict['total_tweets'] += 1

        # nie przetwarzaj tweetów nieposiadających informacji 
        # o lokalizacji
        if not status.user.location:  
            return
            
        # zliczaj wyłącznie tweety z informacją o lokalizacji
        self.counts_dict['locations'] += 1  
        self.tweets_list.append(tweet_data)  # dołącz tweet do listy
        print(f'{status.user.screen_name}: {tweet_data["text"]}\n')

        # gdy przekroczony zostanie limit przetworzonych tweetów
        # (TWEET_LIMIT) zwróć False w celu przerwania strumieniowania,
        # w przeciwnym razie zwróć True.
        return self.counts_dict['locations'] <= self.TWEET_LIMIT



##########################################################################
# (C) Copyright 2019 by Deitel & Associates, Inc. and                    #
# Pearson Education, Inc. All Rights Reserved.                           #
#                                                                        #
# DISCLAIMER: The authors and publisher of this book have used their     #
# best efforts in preparing the book. These efforts include the          #
# development, research, and testing of the theories and programs        #
# to determine their effectiveness. The authors and publisher make       #
# no warranty of any kind, expressed or implied, with regard to these    #
# programs or to the documentation contained in these books. The authors #
# and publisher shall not be liable in any event for incidental or       #
# consequential damages in connection with, or arising out of, the       #
# furnishing, performance, or use of these programs.                     #
##########################################################################
