"""Python. Programowanie funkcyjne, wydanie 3

Rozdział 15., Konfiguracja testów
"""
from collections.abc import Iterator
from pathlib import Path
import subprocess
import time

import pytest

def pytest_configure(config: pytest.Config) -> None:
    config.addinivalue_line(
        "markers", "server_file(file): mark test to use a specific file"
    )

@pytest.fixture
def running_server(request: pytest.FixtureRequest) -> Iterator[Path]:
    """
    Uruchamia moduł o podanej nazwie jako serwer WSGI.
    Idealnie w postaci scope="module"
    """
    server_file = request.node.get_closest_marker("server_file").args[0]

    log_path = Path.cwd() / "server.log"
    with log_path.open("w") as log_file:
        log_file.write(f"Uruchamiam serwer {server_file!r}...\n")
        with subprocess.Popen(
            ["python", server_file],
            stdout=log_file,
            stderr=subprocess.STDOUT,
            text=True
        ) as server:
            time.sleep(1)  # Czas na uruchomienie potomka
            log_file.write(f"Awaria uruchamiania? {server.poll()=}\n")
            log_file.flush()

            yield log_path  # Wait while tests run

            log_file.write(f"Zabijam {server.pid=}\n")
            server.kill()
            time.sleep(0.5)  # Oczekiwanie na zakończenie
            log_file.write(f"Finalizacja {server.poll()=}\n")
            log_file.flush()
