#!/usr/bin/env python

import os
from collections import OrderedDict

import commands


def print_bookmarks(bookmarks):
    for bookmark in bookmarks:
        print('\t'.join(
            str(field) if field else ''
            for field in bookmark
        ))


class Option:
    def __init__(self, name, command, prep_call=None):
        self.name = name
        self.command = command
        self.prep_call = prep_call

    def _handle_message(self, message):
        if isinstance(message, list):
            print_bookmarks(message)
        else:
            print(message)

    def choose(self):
        data = self.prep_call() if self.prep_call else None
        message = self.command.execute(data) if data else self.command.execute()
        self._handle_message(message)

    def __str__(self):
        return self.name


def clear_screen():
    clear = 'cls' if os.name == 'nt' else 'clear'
    os.system(clear)


def print_options(options):
    for shortcut, option in options.items():
        print(f'({shortcut}) {option}')
    print()


def option_choice_is_valid(choice, options):
    return choice in options or choice.upper() in options


def get_option_choice(options):
    choice = input('Wybierz opcję: ')
    while not option_choice_is_valid(choice, options):
        print('Nieprawidłowy wybór')
        choice = input('Wybierz opcję: ')
    return options[choice.upper()]


def get_user_input(label, required=True):
    value = input(f'{label}: ') or None
    while required and not value:
        value = input(f'{label}: ') or None
    return value


def get_new_bookmark_data():
    return {
        'title': get_user_input('Tytuł'),
        'url': get_user_input('URL'),
        'notes': get_user_input('Notatki', required=False),
    }


def get_bookmark_id_for_deletion():
    return get_user_input('Wprowadź ID zakładki do usunięcia')


def get_github_import_options():  # <1>
    return {
        'github_username': get_user_input('Nazwa użytkownika GitHub'),
        'preserve_timestamps':  # <2>
            get_user_input(
                'Zachować znaczniki czasu? [Y/n]',
                required=False
            ) in {'Y', 'y', None},  # <3>
    }


def get_new_bookmark_info():
    bookmark_id = get_user_input('Wprowadź ID zakładki do edycji')
    field = get_user_input('Wybierz wartość do edycji (Tytuł, URL, Notatki)')
    new_value = get_user_input(f'Enter the new value for {field}')
    return {
        'id': bookmark_id,
        'update': {field: new_value},
    }


def loop():
    clear_screen()

    options = OrderedDict({
        'A': Option('Dodaj zakładkę', commands.AddBookmarkCommand(), prep_call=get_new_bookmark_data),
        'B': Option('Lista zakładek względem daty', commands.ListBookmarksCommand()),
        'T': Option('Lista zakładek względem tytułu', commands.ListBookmarksCommand(order_by='title')),
        'E': Option('Edytuj zakładkę', commands.EditBookmarkCommand(), prep_call=get_new_bookmark_info),
        'D': Option('Usuń zakładkę', commands.DeleteBookmarkCommand(), prep_call=get_bookmark_id_for_deletion),
        'G': Option(  # <4>
            'Importuj gwiazdki GitHub',
            commands.ImportGitHubStarsCommand(),
            prep_call=get_github_import_options
        ),
        'Q': Option('Wyjdź', commands.QuitCommand()),
    })
    print_options(options)

    chosen_option = get_option_choice(options)
    clear_screen()
    chosen_option.choose()

    _ = input('Naciśnij ENTER aby powrócić do menu')


if __name__ == '__main__':
    commands.CreateBookmarksTableCommand().execute()

    while True:
        loop()
