#!/usr/bin/env python

import os
from collections import OrderedDict

import commands


def format_bookmark(bookmark):
    return '\t'.join(
        str(field) if field else ''
        for field in bookmark
    )


class Option:
    def __init__(self, name, command, prep_call=None, success_message='{result}'):  # <1>
        self.name = name
        self.command = command
        self.prep_call = prep_call
        self.success_message = success_message  # <2>

    def choose(self):
        data = self.prep_call() if self.prep_call else None
        success, result = self.command.execute(data)  # <3>

        formatted_result = ''

        if isinstance(result, list):  # <4>
            for bookmark in result:
                formatted_result += '\n' + format_bookmark(bookmark)
        else:
            formatted_result = result

        if success:
            print(self.success_message.format(result=formatted_result))  # <5>

    def __str__(self):
        return self.name


def clear_screen():
    clear = 'cls' if os.name == 'nt' else 'clear'
    os.system(clear)


def print_options(options):
    for shortcut, option in options.items():
        print(f'({shortcut}) {option}')
    print()


def option_choice_is_valid(choice, options):
    return choice in options or choice.upper() in options


def get_option_choice(options):
    choice = input('Wybierz opcję: ')
    while not option_choice_is_valid(choice, options):
        print('Nieprawidłowy wybór')
        choice = input('Wybierz opcję: ')
    return options[choice.upper()]


def get_user_input(label, required=True):
    value = input(f'{label}: ') or None
    while required and not value:
        value = input(f'{label}: ') or None
    return value


def get_new_bookmark_data():
    return {
        'title': get_user_input('Tytuł'),
        'url': get_user_input('URL'),
        'notes': get_user_input('Notatki', required=False),
    }


def get_bookmark_id_for_deletion():
    return get_user_input('Wprowadź ID zakładki do usunięcia')


def get_github_import_options():
    return {
        'github_username': get_user_input('Nazwa użytkownika GitHub'),
        'preserve_timestamps':
            get_user_input(
                'Zachować znaczniki czasu? [Y/n]',
                required=False
            ) in {'Y', 'y', None},
    }


def get_new_bookmark_info():
    bookmark_id = get_user_input('Wprowadź ID zakładki do edycji')
    field = get_user_input('Wybierz wartość do edycji (Tytuł, URL, Notatki)')
    new_value = get_user_input(f'Wprowadź wartość dla {field}')
    return {
        'id': bookmark_id,
        'update': {field: new_value},
    }


def loop():
    clear_screen()

    options = OrderedDict({
        'A': Option(
            'Dodaj zakładkę',
            commands.AddBookmarkCommand(),
            prep_call=get_new_bookmark_data,
            success_message='Dodano zakładkę!',  # <6>
        ),
        'B': Option(
            'Lista zakładek względem daty',
            commands.ListBookmarksCommand(),  # <7>
        ),
        'T': Option(
            'Lista zakładek względem tytułu',
            commands.ListBookmarksCommand(order_by='title'),
        ),
        'E': Option(
            'Edytuj zakładkę',
            commands.EditBookmarkCommand(),
            prep_call=get_new_bookmark_info,
            success_message='Zaktualizowano zakładkę!'
        ),
        'D': Option(
            'Usuń zakładkę',
            commands.DeleteBookmarkCommand(),
            prep_call=get_bookmark_id_for_deletion,
            success_message='Usunięto zakłądkę!',
        ),
        'G': Option(
            'Importuj gwiazdki GitHub',
            commands.ImportGitHubStarsCommand(),
            prep_call=get_github_import_options,
            success_message=f'Importowano {result} zakładek z repozytoriów oznaczonych gwiazdką!',  # <8>
        ),
        'Q': Option(
            'Quit',
            commands.QuitCommand()
        ),
    })
    print_options(options)

    chosen_option = get_option_choice(options)
    clear_screen()
    chosen_option.choose()

    _ = input('Naciśnij ENTER aby powrócić do menu')


if __name__ == '__main__':
    while True:
        loop()
