import unittest

from cart import ShoppingCart
from product import Product


class ShoppingCartTestCase(unittest.TestCase):  # <1>
    def test_add_and_remove_product(self):
        cart = ShoppingCart()  # <2>
        product = Product('buty', 'S', 'niebieski')  # <3>

        cart.add_product(product)  # <4>
        cart.remove_product(product)  # <5>

        self.assertDictEqual({}, cart.products)  # <6>


# Full test suite for product and shopping cart
class ProductTestCase(unittest.TestCase):
    def test_transform_name_for_sku(self):
        small_black_shoes = Product('buty', 'S', 'czarny')
        self.assertEqual(
            'BUTY',
            small_black_shoes.transform_name_for_sku(),
        )

    def test_transform_color_for_sku(self):
        small_black_shoes = Product('buty', 'S', 'czarny')
        self.assertEqual(
            'CZARNY',
            small_black_shoes.transform_color_for_sku(),
        )

    def test_generate_sku(self):
        small_black_shoes = Product('buty', 'S', 'czarny')
        self.assertEqual(
            'BUTY-S-CZARNY',
            small_black_shoes.generate_sku(),
        )


class ShoppingCartTestCase(unittest.TestCase):
    def test_cart_initially_empty(self):
        cart = ShoppingCart()
        self.assertDictEqual({}, cart.products)

    def test_add_product(self):
        cart = ShoppingCart()
        product = Product('buty', 'S', 'niebieski')

        cart.add_product(product)

        self.assertDictEqual({'BUTY-S-NIEBIESKI': {'quantity': 1}}, cart.products)

    def test_add_two_of_a_product(self):
        cart = ShoppingCart()
        product = Product('buty', 'S', 'niebieski')

        cart.add_product(product, quantity=2)

        self.assertDictEqual({'BUTY-S-NIEBIESKI': {'quantity': 2}}, cart.products)

    def test_add_two_different_products(self):
        cart = ShoppingCart()
        product_one = Product('buty', 'S', 'niebieski')
        product_two = Product('koszula', 'S', 'szary')

        cart.add_product(product_one)
        cart.add_product(product_two)

        self.assertDictEqual(
            {
                'BUTY-S-NIEBIESKI': {'quantity': 1},
                'KOSZULA-M-SZARY': {'quantity': 1}
            },
            cart.products
        )

    def test_add_and_remove_product(self):
        cart = ShoppingCart()
        product = Product('buty', 'S', 'niebieski')

        cart.add_product(product)
        cart.remove_product(product)

        self.assertDictEqual({}, cart.products)

    def test_remove_too_many_products(self):
        cart = ShoppingCart()
        product = Product('buty', 'S', 'niebieski')

        cart.add_product(product)
        cart.remove_product(product, quantity=2)

        self.assertDictEqual({}, cart.products)


# Using pytest
class TestProduct:  # <1>
    def test_transform_name_for_sku(self):
        small_black_shoes = Product('buty', 'S', 'czarny')
        assert small_black_shoes.transform_name_for_sku() == 'BUTY'  # <2>

    def test_transform_color_for_sku(self):
        small_black_shoes = Product('buty', 'S', 'czarny')
        assert small_black_shoes.transform_color_for_sku() == 'CZARNY'

    def test_generate_sku(self):
        small_black_shoes = Product('buty', 'S', 'czarny')
        assert small_black_shoes.generate_sku() == 'BUTY-S-CZARNY'
