'''
Ten skrypt wczytuje plik CSV i generuje inny plik CSV ze zagregowanymi danymi.
'''
import csv
import argparse
import delorean
from decimal import Decimal


def parse_iso(timestamp):
    # Przetwarzanie daty w formacie ISO.
    total = delorean.parse(timestamp, dayfirst=False)
    # Zachowywanie samej daty.
    return total.date


def line(date, total_usd, number):
    data = {
        'DATA': date,
        'USD W SUMIE': total_usd,
        'LICZBA': number,
        # Zakrąglanie do dwóch miejsc po przecinku.
        'ŚREDNIA': round(total_usd / number, 2),
    }
    return data


def calculate_results(reader):
    result = []
    last_date = None
    total_usd = 0
    number = 0

    for row in reader:
        date = parse_iso(row['CZAS_STANDARDOWY'])
        if not last_date:
            last_date = date

        if last_date < date:
            # Nowy dzień!
            result.append(line(last_date, total_usd, number))
            total_usd = 0
            number = 0
            last_date = date

        number += 1
        total_usd += Decimal(row['USD'])

    # Ostateczne wyniki.
    result.append(line(date, total_usd, number))
    return result


def main(input_file, output_file):
    reader = csv.DictReader(input_file)
    result = calculate_results(reader)

    # Zapisywanie pliku w formacie csv.
    header = result[0].keys()
    writer = csv.DictWriter(output_file, fieldnames=header)
    writer.writeheader()
    writer.writerows(result)


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument(dest='input', type=argparse.FileType('r'),
                        help='Plik wejściowy')
    parser.add_argument(dest='output', type=argparse.FileType('w'),
                        help='Plik wyjściowy')
    args = parser.parse_args()
    main(args.input, args.output)
