//: Playground - noun: a place where people can play

import Cocoa

struct StackGenerator<T>: IteratorProtocol {
    var stack: Stack<T>
    
    mutating func next() -> T? {
        return stack.pop()
    }
}

struct Stack<Element>: Sequence, IteratorProtocol {
    var items = [Element]()
    
    mutating func push(newItem: Element) {
        items.append(newItem)
    }
    
    mutating func pop() -> Element? {
        guard !items.isEmpty else {
            return nil
        }
        return items.removeLast()
    }
    
    func map<U>(f: (Element) -> U) -> Stack<U> {
        var mappedItems = [U]()
        for item in items {
            mappedItems.append(f(item))
        }
        return Stack<U>(items: mappedItems)
    }
    
    func generate() -> StackGenerator<Element> {
        return StackGenerator(stack: self)
    }
    
    mutating func next() -> Element? {
        guard !items.isEmpty else {
            return nil
        }
        return items.removeLast()
    }
}

var intStack = Stack<Int>()
intStack.push(newItem:1)
intStack.push(newItem:2)
var doubledStack = intStack.map(f: { 2 * $0 })

print(intStack.pop()) // Dane wyjściowe to Optional(2)
print(intStack.pop()) // Dane wyjściowe to Optional(1)
print(intStack.pop()) // Dane wyjściowe to nil

print(doubledStack.pop()) // Dane wyjściowe to Optional(4).
print(doubledStack.pop()) // Dane wyjściowe to Optional(2).


func myMap<T,U>(items: [T], f: (T) -> (U)) -> [U] {
    var result = [U]()
    for item in items {
        result.append(f(item))
    }
    return result
}

let strings = ["jeden", "dwa", "trzy"]
let stringLengths = myMap(items: strings) { $0.characters.count }
print(stringLengths) // Dane wyjściowe to [5, 3, 4].

func checkIfEqual<T: Equatable>(first: T, _ second: T) -> Bool {
    return first == second
}

print(checkIfEqual(first: 1, 1))
print(checkIfEqual(first: "ciąg tekstowy", "ciąg tekstowy"))
print(checkIfEqual(first: "ciąg tekstowy", "inny ciąg tekstowy"))

func checkIfDescriptionsMatch<T: CustomStringConvertible, U: CustomStringConvertible>(
    first: T, _ second: U) -> Bool {
    return first.description == second.description
}

print(checkIfDescriptionsMatch(first: Int(1), UInt(1)))
print(checkIfDescriptionsMatch(first: 1, 1.0))
print(checkIfDescriptionsMatch(first: Float(1.0), Double(1.0)))

var myStack = Stack<Int>()
myStack.push(newItem:10)
myStack.push(newItem:20)
myStack.push(newItem:30)

var myStackGenerator = StackGenerator(stack: myStack)
while let value = myStackGenerator.next() {
    print("Otrzymana wartość to \(value).")
}

for value in myStack.items {
    print("Wartość pochodząca z pętli for-in: \(value).")
}

func pushItemsOntoStack<Element, S: Sequence>
    (stack: inout Stack<Element>, fromSequence sequence: S)
    where S.Iterator.Element == Element {
    for item in sequence {
        stack.push(newItem:item)
    }
}

pushItemsOntoStack(stack: &myStack, fromSequence: [1, 2, 3])
for value in myStack.items {
    print("Po umieszczeniu na stosie: otrzymano wartość \(value).")
}

var myOtherStack = Stack<Int>()
pushItemsOntoStack(stack: &myOtherStack, fromSequence: [1, 2, 3])
pushItemsOntoStack(stack: &myStack, fromSequence: myOtherStack)
for value in myStack.items {
    print("Po umieszczeniu elementów na stosie otrzymujemy wartość \(value).")
}

