//: Playground - noun: a place where people can play

import Cocoa

protocol TabularDataSource {
    var numberOfRows: Int { get }
    var numberOfColumns: Int { get }
    
    func labelForRow(row: Int) -> String
    func labelForColumn(column: Int) -> String
    
    func itemForRow(row: Int, column: Int) -> Int
}

func padding(amount: Int) -> String {
    var paddingString = ""
    for _ in 0 ..< amount {
        paddingString += " "
    }
    return paddingString
}

func printTable(dataSource: TabularDataSource & CustomStringConvertible) {
    print("Tabela: \(dataSource.description)")
    // Utworzenie tablic etykiet dla wierszy i kolumn.
    let rowLabels = (0 ..< dataSource.numberOfRows).map { dataSource.labelForRow(row: $0) }
    let columnLabels = (0 ..< dataSource.numberOfColumns).map {
        dataSource.labelForColumn(column: $0)
    }

    // Utworzenie tablicy szerokości każdego wiersza etykiety.
    let rowLabelWidths = rowLabels.map { $0.characters.count }
    
    // Ustalenie długości najdłuższego wiersza etykiety.
    guard let maxRowLabelWidth = rowLabelWidths.max() else {
        return
    }
    
    // Utworzenie pierwszego wiersza zawierającego nagłówki kolumn.
    var firstRow: String = padding(amount: maxRowLabelWidth) + " |"
    
    // Monitorowanie długości każdej kolumny.
    var columnWidths = [Int]()
    
    for columnLabel in columnLabels {
        let columnHeader = " \(columnLabel) |"
        firstRow += columnHeader
        columnWidths.append(columnHeader.characters.count)
    }
    print(firstRow)
    
    for i in 0 ..< dataSource.numberOfRows {
        // Dopełnienie etykiety spacjami, aby wszystkie miały tę samą długość
        let paddingAmount = maxRowLabelWidth - rowLabelWidths[i]
        var out = rowLabels[i] + padding(amount: paddingAmount) + " |"
        
        // Dołączenie każdego elementu w tym wierszu do naszego ciągu tekstowego.
        for j in 0 ..< dataSource.numberOfColumns {
            let item = dataSource.itemForRow(row: i, column: j)
            let itemString = " \(item) |"
            let paddingAmount = columnWidths[j] - itemString.characters.count
            out += padding(amount: paddingAmount) + itemString
        }
        
        // Gotowe, można wyświetlić zawartość!
        print(out)
    }
}

struct Person {
    let name: String
    let age: Int
    let yearsOfExperience: Int
}

struct Department: TabularDataSource, CustomStringConvertible {
    let name: String
    var people = [Person]()
    
    var description: String {
        return "Departament (\(name))"
    }
    
    init(name: String) {
        self.name = name
    }
    
    mutating func addPerson(person: Person) {
        people.append(person)
    }
    
    var numberOfRows: Int {
        return people.count
    }
    
    var numberOfColumns: Int {
        return 2
    }
    
    func labelForRow(row: Int) -> String {
        return people[row].name
    }
    
    func labelForColumn(column: Int) -> String {
        switch column {
        case 0: return "Wiek"
        case 1: return "Lata doświadczenia"
        default: fatalError("Invalid column!")
        }
    }
    
    func itemForRow(row: Int, column: Int) -> Int {
        let person = people[row]
        switch column {
        case 0: return person.age
        case 1: return person.yearsOfExperience
        default: fatalError("Nieprawidłowa kolumna!")
        }
    }
}

var department = Department(name: "Inżynieria")
department.addPerson(person: Person(name: "Jan", age: 30, yearsOfExperience: 6))
department.addPerson(person: Person(name: "Karen", age: 40, yearsOfExperience: 18))
department.addPerson(person: Person(name: "Franek", age: 50, yearsOfExperience: 20))

printTable(dataSource: department)
