"""
Programowanie obiektowe w Pythonie 3

Rozdział 9., Łańcuchy, serializacja i ścieżki do plików
"""

from math import cos, radians, hypot, pi


def distance(lat1: float, lon1: float, lat2: float, lon2: float) -> float:
    """
    >>> annapolis = (38.9784, 76.4922)
    >>> saint_michaels = (38.7854, 76.2233)
    >>> round(distance(*annapolis, *saint_michaels), 9)
    17.070608794
    """
    d_lat = radians(lat2) - radians(lat1)
    d_lon = min(
        (radians(lon2) - radians(lon1)) % (2 * pi),
        (radians(lon1) - radians(lon2)) % (2 * pi),
    )
    R = 60 * 180 / pi
    d = hypot(R * d_lat, R * cos(radians(lat1)) * d_lon)
    return d


test_distance = """

>>> annapolis = (38.9784, 76.4922)
>>> saint_michaels = (38.7854, 76.2233)
>>> oxford = (38.6865, 76.1716)
>>> cambridge = (38.5632, 76.0788)

>>> legs = [
...     ("do st.michaels", annapolis, saint_michaels),
...     ("do oxford", saint_michaels, oxford),
...     ("do cambridge", oxford, cambridge),
...     ("powrót", cambridge, annapolis),
... ]

>>> speed = 5
>>> fuel_per_hr = 2.2
>>> for name, start, end in legs:
...     d = distance(*start, *end)
...     print(name, d, d/speed, d/speed*fuel_per_hr)
do st.michaels 17.070608794397305 3.4141217588794612 7.511067869534815
do oxford 6.407736547720565 1.281547309544113 2.8194040809970486
do cambridge 8.580230239760066 1.7160460479520132 3.7753013054944295
powrót 31.571582240989173 6.314316448197834 13.891496186035237


>>> speed = 5
>>> fuel_per_hr = 2.2
>>> print(f"{'etap':16s} {'dyst':5s} {'czas':4s} {'paliwo':6s}")
etap             dyst  czas paliwo
>>> for name, start, end in legs:
...     d = distance(*start, *end)
...     print(
...         f"{name:16s} {d:5.2f} {d/speed:4.1f} "
...         f"{d/speed*fuel_per_hr:4.0f}"
...     )
do st.michaels   17.07  3.4    8
do oxford         6.41  1.3    3
do cambridge      8.58  1.7    4
powrót           31.57  6.3   14



"""

__test__ = {name: case for name, case in globals().items() if name.startswith("test_")}
