"""
Programowanie obiektowe w Pythonie 3

Rozdział 4., Oczekując nieoczekiwanego
"""
from __future__ import annotations


def divide_with_exception(dividend: int, divisor: int) -> None:
    try:
        print(f"{dividend / divisor=}")
    except ZeroDivisionError:
        print("Nie można dzielić przez zero")


def divide_with_if(dividend: int, divisor: int) -> None:
    if divisor == 0:
        print("Nie można dzielić przez zero")
    else:
        print(f"{dividend / divisor=}")


class ItemType:
    def __init__(self, name: str) -> None:
        self.name = name
        self.on_hand = 0


class OutOfStock(Exception):
    pass


class InvalidItemType(Exception):
    pass


class Inventory:
    def __init__(self, stock: list[ItemType]) -> None:
        pass

    def lock(self, item_type: ItemType) -> None:
        """Punkt wejścia do kontekstu.
        Blokuje typ produktu, tak by nikt inny nie mógł 
        nic z nim robić kiedy my go używamy."""
        pass

    def unlock(self, item_type: ItemType) -> None:
        """Punkt wyjścia z kontekstu.
        Odblokowujemy typ produktu."""
        pass

    def purchase(self, item_type: ItemType) -> int:
        """Jeśli produkt nie jest zablokowany, zgłaszamy
        ValueError, coś jest nie w porządku.
        Jeśli item_type nie istnieje, zgłaszamy
           wyjątek InvalidItemType.
        Jeśli produktu nie ma w magazynie, zgłaszamy 
           wyjątek OutOfStock.
        Jeśli produkt jest dostępny, odejmujemy 
          jeden egzemplarz i zwracamy liczbę pozostałych egzemplarzy
        """
        # Fikcyjne wyniki.
        if item_type.name == "Gadżet":
            raise OutOfStock(item_type)
        elif item_type.name == "Gizmo":
            return 42
        else:
            raise InvalidItemType(item_type)


test_inventory = """
>>> widget = ItemType("Gadżet")
>>> gadget = ItemType("Gizmo")
>>> inv = Inventory([widget, gadget])

>>> item_to_buy = widget
>>> inv.lock(item_to_buy)
>>> try:
...     num_left = inv.purchase(item_to_buy)
... except InvalidItemType:
...     print(f"Przykro nan, ale nie sprzedajemy {item_to_buy.name}.")
... except OutOfStock:
...     print("Przykro nam, ale tego produktu nie ma w magazynie.")
... else:
...     print(f"Dokonano zakupu. Zostało {num_left} ezgemplarzy {item_to_buy.name}.")
... finally:
...     inv.unlock(item_to_buy)
...
Przykro nam, ale tego produktu nie ma w magazynie.

>>> item_to_buy = gadget
>>> inv.lock(item_to_buy)
>>> try:
...     num_left = inv.purchase(item_to_buy)
... except InvalidItemType:
...     print(f"Przykro nan, ale nie sprzedajemy {item_to_buy.name}.")
... except OutOfStock:
...     print("Przykro nam, ale tego produktu nie ma w magazynie.")
... else:
...     print(f"Dokonano zakupu. Zostało {num_left} ezgemplarzy {item_to_buy.name}.")
... finally:
...     inv.unlock(item_to_buy)
...
Dokonano zakupu. Zostało 42 ezgemplarzy Gizmo.

>>> item_to_buy = ItemType("Sprocket")
>>> inv.lock(item_to_buy)
>>> try:
...     num_left = inv.purchase(item_to_buy)
... except InvalidItemType:
...     print(f"Przykro nan, ale nie sprzedajemy {item_to_buy.name}.")
... except OutOfStock:
...     print("Przykro nam, ale tego produktu nie ma w magazynie.")
... else:
...     print(f"Dokonano zakupu. Zostało {num_left} ezgemplarzy {item_to_buy.name}.")
... finally:
...     inv.unlock(item_to_buy)
...
Przykro nan, ale nie sprzedajemy Sprocket.

"""

__test__ = {name: case for name, case in globals().items() if name.startswith("test_")}


if __name__ == "__main__":
    # Porównanie wydajności dwóch podejść: "łatwiej prosić o wybaczenie niż o pozwolenie"
    # oraz "nie rób nic pochopnie".
    import timeit
    import sys

    eafp_pure_happy = timeit.timeit(
        setup="from manufacturing import divide_with_exception; import io; import sys; sys.stdout=io.StringIO()",
        stmt="divide_with_exception(355, 113)",
    )
    eafp_pure_sad = timeit.timeit(
        setup="from manufacturing import divide_with_exception; import io; import sys; sys.stdout=io.StringIO()",
        stmt="divide_with_exception(355, 0)",
    )
    eafp_90_10 = timeit.timeit(
        setup="from manufacturing import divide_with_exception; import io; import sys; sys.stdout=io.StringIO()",
        stmt="for i in range(10): divide_with_exception(355, i)",
        number=100_000,
    )
    eafp_99_1 = timeit.timeit(
        setup="from manufacturing import divide_with_exception; import io; import sys; sys.stdout=io.StringIO()",
        stmt="for i in range(100): divide_with_exception(355, i)",
        number=10_000,
    )

    lbyl_pure_happy = timeit.timeit(
        setup="from manufacturing import divide_with_if; import io; import sys; sys.stdout=io.StringIO()",
        stmt="divide_with_if(355, 113)",
    )
    lbyl_pure_sad = timeit.timeit(
        setup="from manufacturing import divide_with_if; import io; import sys; sys.stdout=io.StringIO()",
        stmt="divide_with_if(355, 0)",
    )
    lbyl_90_10 = timeit.timeit(
        setup="from manufacturing import divide_with_if; import io; import sys; sys.stdout=io.StringIO()",
        stmt="for i in range(10): divide_with_if(355, i)",
        number=100_000,
    )
    lbyl_99_1 = timeit.timeit(
        setup="from manufacturing import divide_with_if; import io; import sys; sys.stdout=io.StringIO()",
        stmt="for i in range(100): divide_with_if(355, i)",
        number=10_000,
    )

    sys.stdout = sys.__stdout__
    print("Łatwiej prosić o wybaczenie niż o pozwolenie")
    print(f"Czyste szczęście       {eafp_pure_happy:.3f}")
    print(f"Wyjątkowo zadowolony   {eafp_pure_sad:.3f}")
    print(f" 90% zadowolenia       {eafp_90_10:.3f}")
    print(f"  1% zadowolenia       {eafp_99_1:.3f}")

    print("Nie rób nic pochopnie")
    print(f"Czyste szczęście       {lbyl_pure_happy:.3f}")
    print(f"Wyjątkowo zadowolony   {lbyl_pure_sad:.3f}")
    print(f" 90% zadowolenia       {lbyl_90_10:.3f}")
    print(f"  1% zadowolenia       {lbyl_99_1:.3f}")
