"""
Programowanie obiektowe w Pythonie 3

Rozdział 8., Łączenie programowania obiektowego i funkcyjnego
"""
from typing import List, Optional, Type, Literal
from types import TracebackType


class StringJoiner(List[str]):
    def __enter__(self) -> "StringJoiner":
        return self
    
    def __exit__(
        self,
        exc_type: Optional[Type[BaseException]],
        exc_val: Optional[BaseException],
        exc_tb: Optional[TracebackType],
    ) -> Literal[False]:  # Optional[bool]:
        self.result = "".join(self)
        return False


test_joiner = """
>>> with StringJoiner("Witaj") as sj:
...     sj.append(", ")
...     sj.extend("świecie")
...     sj.append("!")
>>> sj.result
'Witaj, świecie!'
"""

test_joiner_exception = """
>>> with StringJoiner("Wyniki") as sj:
...     sj.append(" ")
...     sj.extend("częściowe")
...     sj.append(str(2 / 0))
...     sj.extend("nawet jeśli zgłoszono wyjątek")
Traceback (most recent call last):
  ...
  File "<doctest examples.md[60]>", line 3, in <module>
    sj.append(str(2 / 0))
ZeroDivisionError: division by zero
>>> sj.result
'Wyniki częściowe'
"""

from contextlib import contextmanager
from typing import List, Any, Iterator


class StringJoiner2(List[str]):
    def __init__(self, *args: str) -> None:
        super().__init__(*args)
        self.result = "".join(self)


@contextmanager
def joiner(*args: Any) -> Iterator[StringJoiner2]:
    string_list = StringJoiner2(*args)
    try:
        yield string_list
    finally:
        string_list.result = "".join(string_list)


test_joiner2 = """
>>> with joiner("Witaj") as sj:
...     sj.append(", ")
...     sj.extend("świecie")
...     sj.append("!")
>>> sj.result
'Witaj, świecie!'
"""

test_joiner2_exception = """
>>> with joiner("Wyniki") as sj:
...     sj.append(" ")
...     sj.extend("częściowe")
...     sj.append(str(2 / 0))
...     sj.extend("Nawet jeśli zostanie zgłoszony wyjątek")
Traceback (most recent call last):
  ...
  File "<doctest examples.md[60]>", line 3, in <module>
    sj.append(str(2 / 0))
ZeroDivisionError: division by zero
>>> sj.result
'Wyniki częściowe'
"""


__test__ = {name: case for name, case in globals().items() if name.startswith("test_")}


if __name__ == "__main__":
    results = str(2 ** 2048)
    with open("big_number.txt", "w") as output:
        output.write("# Bardzo duża liczba\n")
        output.writelines([f"{len(results)}\n", f"{results}\n"])
    with open("big_number.txt") as input:
        for line in input:
            print(line)
