# Programowanie obiektowe w Pythonie 3

Rozdział 10., Wzorzec Iterator

## Krótko o wzorcach projektowych

## Iteratory

## Listy składane

### Wyrażenia list składanych

```python

>>> input_strings = ["1", "5", "28", "131", "3"]
 
>>> output_integers = [] 
>>> for num in input_strings: 
...    output_integers.append(int(num)) 
>>> output_integers
[1, 5, 28, 131, 3]

```

```python

>>> input_strings = ["1", "5", "28", "131", "3"]
 
>>> output_integers = [int(num) for num in input_strings] 
>>> output_integers
[1, 5, 28, 131, 3]

```

```python
>>> input_strings = ["1", "5", "28", "131", "3"]
>>> output_integers = [int(num) for num in input_strings if len(num) < 3]
>>> output_integers
[1, 5, 28, 3]

```

Uwaga: W razie korzystania z systemu Windows w wywołaniu assert należy zmienić klasę z ```Path``` na ```WindowsPath```.

```python
>>> from pathlib import Path, PosixPath, WindowsPath

>>> chapter = Path.cwd()
>>> paths = [path.relative_to(chapter) 
...     for path in chapter.glob("src/*.py") 
...     if ">>>" in path.read_text()
... ]
>>> paths.sort()
>>> import sys

#...     paths == [WindowsPath('src/iterator_protocol.py'), WindowsPath('src/log_analysis.py'), WindowsPath('src/model.py')]
#...     paths == [Path('src/iterator_protocol.py'), Path('src/log_analysis.py'), Path('src/model.py')]

>>> assert (
...     paths == [WindowsPath('src/iterator_protocol.py'), WindowsPath('src/log_analysis.py'), WindowsPath('src/model.py')]
... ), f"Ścieżka {paths!r} jest nieprawidłowa dla platformy {sys.platform!r}"

>>> source_path = Path('src') / 'iterator_protocol.py'
>>> with source_path.open() as source:
...     examples = [line.rstrip() 
...         for line in source 
...         if ">>>" in line]
>>> examples
[">>> iterable = CapitalIterable('szybki rudy lis przeskoczył leniwego psa')", '>>> iterator = iter(iterable)', '>>> while True:', '>>> for i in iterable:', '>>> iterator = iter(iterable)', '>>> for i in iter(iterator):']

>>> source_path = Path('src') / 'iterator_protocol.py'
>>> with source_path.open() as source:
...     examples = [(number, line.rstrip()) 
...         for number, line in enumerate(source, start=1) 
...         if ">>>" in line]
>>> examples
[(35, ">>> iterable = CapitalIterable('szybki rudy lis przeskoczył leniwego psa')"), (36, '>>> iterator = iter(iterable)'), (37, '>>> while True:'), (50, '>>> for i in iterable:'), (60, '>>> iterator = iter(iterable)'), (61, '>>> for i in iter(iterator):')]

>>> import doctest
>>> import iterator_protocol
>>> test_finder = doctest.DocTestFinder()
>>> [test.name for test in test_finder.find(iterator_protocol)]
['iterator_protocol', 'iterator_protocol.__test__.test_iterable']


```

### Wyrażenia zbiorów i słowników składanych

```python

>>> from typing import NamedTuple

>>> class Book(NamedTuple):
...     author: str
...     title: str
...     genre: str

>>> books = [
...     Book("Pratchett", "Straż nocna", "fantastyka"),
...     Book("Pratchett", "Złodziej czasu", "fantastyka"),
...     Book("Le Guin", "Najdalszy brzeg", "science fiction"),
...     Book("Le Guin", "Czarnoksiężnik z Archipelagu", "fantastyka"),
...     Book("Jemisin", "Mroczne słońce", "fantastyka"),
...     Book("Turner", "Złodziej", "fantastyka"),
...     Book("Phillips", "Preston Diamond", "western"),
...     Book("Simmons", "Hyperion", "science fiction"),
... ]

>>> fantasy_authors = {b.author for b in books if b.genre == "fantastyka"}
>>> fantasy_authors == {'Pratchett', 'Le Guin', 'Turner', 'Jemisin'}
True

>>> fantasy_titles = {b.title: b for b in books if b.genre == "fantastyka"}
>>> fantasy_titles
{'Straż nocna': Book(author='Pratchett', title='Straż nocna', genre='fantastyka'), 'Złodziej czasu': Book(author='Pratchett', title='Złodziej czasu', genre='fantastyka'), 'Czarnoksiężnik z Archipelagu': Book(author='Le Guin', title='Czarnoksiężnik z Archipelagu', genre='fantastyka'), 'Mroczne słońce': Book(author='Jemisin', title='Mroczne słońce', genre='fantastyka'), 'Złodziej': Book(author='Turner', title='Złodziej', genre='fantastyka')}
>>> fantasy_titles['Straż nocna']
Book(author='Pratchett', title='Straż nocna', genre='fantastyka')


```

## Wyrażenia generatorów

```python
>>> from pathlib import Path

>>> full_log_path = Path.cwd() / "data" / "sample.log"
>>> warning_log_path = Path.cwd() / "data" / "warnings.log"

>>> with full_log_path.open() as source:
...     warning_lines = (line for line in source if "WARN" in line)
...     with warning_log_path.open('w') as target:
...         for line in warning_lines:
...             _ = target.write(line)

>>> with warning_log_path.open() as warnings:
...     for line in warnings:
...         print(line.rstrip())
Apr 05, 2021 20:03:53 WARNING This is a warning. It could be serious.
Apr 05, 2021 20:03:59 WARNING Another warning sent.
Apr 05, 2021 20:04:35 WARNING Warnings should be heeded.
Apr 05, 2021 20:04:41 WARNING Watch for warnings.



```
