"""
Programowanie obiektowe w Pythonie 3

Rozdział 12., Zaawansowane wzorce projektowe
"""
from __future__ import annotations
import string
import textwrap
from typing import Union, List, Optional


class ListStyleType:
    @staticmethod
    def format(value: int) -> str:
        return f"{value:d}. "


class Decimal(ListStyleType):
    pass


class DecimalLeadingZero(ListStyleType):
    @staticmethod
    def format(value: int) -> str:
        return f"{value:02d}. "


class LowerAlpha(ListStyleType):
    @staticmethod
    def format(value: int) -> str:
        return f"{string.ascii_lowercase[value-1]}. "


class UpperAlpha(ListStyleType):
    @staticmethod
    def format(value: int) -> str:
        return f"{string.ascii_uppercase[value-1]}. "


test_styles = """
>>> l = ListStyleType()
>>> l.format(1)
'1. '
>>> l = Decimal()
>>> l.format(1)
'1. '
>>> l = DecimalLeadingZero()
>>> l.format(1)
'01. '
>>> l = LowerAlpha()
>>> l.format(1)
'a. '
>>> l = UpperAlpha()
>>> l.format(1)
'A. '


"""


class OrderedList:
    def __init__(
        self,
        *children: "Node",
        formatter: type[ListStyleType] = ListStyleType,
    ) -> None:
        self.formatter = formatter
        self.children = list(children)

    def render(self, prefix: str = "") -> str:
        text = []
        for n, c in enumerate(self.children, start=1):
            text.append(c.render(" " * len(prefix) + self.formatter.format(n)))
        return "\n" + ("\n".join(text))


class ListItem:
    def __init__(self, *children: "Node") -> None:
        self.children = list(children)

    def render(self, prefix: str = "") -> str:
        children_iter = iter(self.children)
        first = next(children_iter)
        first_body = [first.render(prefix)]
        item_body = first_body + [c.render(" " * len(prefix)) for c in children_iter]
        return "".join(item_body)


class Text:
    def __init__(self, content: str = "") -> None:
        self.content = content

    def render(self, prefix: str = "") -> str:
        line_iter = iter(self.content.splitlines())
        first = next(line_iter)
        first_body = [f"{prefix}{first}"]
        item_body = first_body + [f"{' '*len(prefix)}{l}" for l in line_iter]
        return "\n".join(item_body) + "\n"


Node = Union[OrderedList, ListItem, Text]

test_text = """
>>> t = Text("nic specjalnego")
>>> t.render(" +-- ")
' +-- nic specjalnego\\n'

>>> t2 = Text("Wiersz 1\\nWiersz 2")
>>> t2.render(" +-- ")
' +-- Wiersz 1\\n     Wiersz 2\\n'

"""


test_list_item = """
>>> li = ListItem(Text("Nic specjalnego"))
>>> li.render("1. ")
'1. Nic specjalnego\\n'

"""

test_ordered_list = """
>>> ol = OrderedList(ListItem(Text("Element 1")), ListItem(Text("Element 2")))
>>> ol.render()
'\\n1. Element 1\\n\\n2. Element 2\\n'

"""

__test__ = {name: case for name, case in globals().items() if name.startswith("test_")}

if __name__ == "__main__":
    doc = OrderedList(
        ListItem(Text("Liść 1")),
        ListItem(
            Text("Kompozyt 1a\n"),
            OrderedList(
                ListItem(Text("Liść 2a")),
                ListItem(
                    Text("Kompozyt 2b\n"),
                    OrderedList(
                        ListItem(Text("Liść 3a")),
                        ListItem(Text("Liść 3b")),
                        ListItem(Text("Liść 3c")),
                        ListItem(Text("Liść 3d")),
                    ),
                ),
                ListItem(Text("Liść 2c")),
                formatter=LowerAlpha,
            ),
        ),
        ListItem(
            Text("Kompozyt 1b\n"),
            OrderedList(
                ListItem(Text("Liść 2d")),
                ListItem(
                    Text("Kompozyt 2e\n"),
                    OrderedList(
                        ListItem(Text("Liść 3e")),
                        ListItem(Text("Liść 3f")),
                    ),
                ),
                formatter=UpperAlpha,
            ),
        ),
    )
    print(doc.render())
