/**
 * Abstrakcyjna klasa bazowa dla klas MouseInputHandler i TouchInputHandler. Klasa ta nie służy do 
 * tworzenia instancji. Udostępnia ona mechanizmy obsługi wspólne dla zdarzeń dotykowych i myszki.
 *
 * @param element Element drzewa DOM, na którym pracujemy i dla którego będą rejestrowane funkcje nasłuchujące.
 * Podczas tworzenia gier jest to na ogół płótno.
 */
function InputHandlerBase(element) {
    EventEmitter.call(this);

    // Element DOM
    this._element = element;

    this._moving = false;
    this._lastMoveCoordinates = null;
    this._moveThreshold = 10;
    this._stopDomEvents = true;
}

extend(InputHandlerBase, EventEmitter);

_p = InputHandlerBase.prototype;

_p.getMoveTheshold = function() {
    return this._moveThreshold;
};

_p.setMoveThreshold = function(moveThreshold) {
    this._moveThreshold = moveThreshold;
};

_p.getStopDomEvents = function() {
    return this._stopDomEvents;
};

_p.setStopDomEvents = function(stopDomEvents) {
    this._stopDomEvents = stopDomEvents;
};

/**
 * Nasłuchuje zdarzenia typu "down": mousedown i touchstart.
 * @param e Zdarzenie DOM
 */
_p._onDownDomEvent = function(e) {
    // Musimy zapisać współrzędne, aby obsłużyć próg ruchu.
    // Ten punkt może być początkiem ruchu.

    var coords = this._lastMoveCoordinates = this._getInputCoordinates(e);

    // Wygeneruj zdarzenie "down" — wszystkie współrzędne wraz z oryginalnym zdarzeniem DOM są przekazywane do słuchaczy.
    this.emit("down", {x: coords.x, y: coords.y, domEvent: e});

    // Zazwyczaj chcemy zatrzymać przetwarzanie oryginalnych zdarzeń DOM pochodzących z przeglądarki
    this._stopEventIfRequired(e);
};

/** 
 * Nasłuchuje zdarzeń DOM typu up: mouseup i touchend. Zdarzenie touchend
 * nie określa żadnych współrzędnych, dlatego ta funkcja zostanie przesłonięta w klasie TouchInputHandler.
 * @param e — zdarzenie DOM
 */ 

_p._onUpDomEvent = function(e) {
    // Działa tak samo jak _onDownDomEvent. 
    var coords = this._getInputCoordinates(e);
    this.emit("up", {x: coords.x, y: coords.y, moved: this._moving, domEvent: e});
    this._stopEventIfRequired(e);

    // Interakcja została zakończona. Zresetuj flagę.
    this._moving = false;
};

/** 
 * Nasłuchuje zdarzenia "move" drzewa DOM: mousemove i touchmove. Funkcja ta jest nieco bardziej skomplikowana. 
 * Jej zadaniem jest rejestrowanie odległości od czasu ostatniego "ruchu", a ponadto ignoruje ruch 
 * i zdarzenie, jeśli ruch nastąpił w obrębie koła o promieniu _moveThreshold.
 * @param e — zdarzenie DOM
 */ 

_p._onMoveDomEvent = function(e) {
    var coords = this._getInputCoordinates(e);

    // Oblicz deltę
    var deltaX = coords.x - this._lastMoveCoordinates.x;
    var deltaY = coords.y - this._lastMoveCoordinates.y;

    // Sprawdź próg. Jeśli odległość pomiędzy początkowym stuknięciem a aktualnym położeniem jest większa niż wartość progu — potraktuj je jak prawdziwy ruch
    if (!this._moving && Math.sqrt(deltaX*deltaX + deltaY*deltaY) > this._moveThreshold) {
        this._moving = true;
    }

    // Jeśli aktualną interakcją jest ruch (tj. przekroczyliśmy próg), wygeneruj zdarzenie — w przeciwnym razie zignoruj interakcję.
    if (this._moving) {
        this.emit("move", {x: coords.x, y: coords.y, deltaX: deltaX, deltaY: deltaY, domEvent: e});
        this._lastMoveCoordinates = coords;
    }

    this._stopEventIfRequired(e);
};

_p._stopEventIfRequired = function(e) {
    if (this._stopDomEvents) {
        e.stopPropagation();
        e.preventDefault();
    }
};

_p._getInputCoordinates = function(e) {
    var element = this._element;
    var coords = e.targetTouches ? e.targetTouches[0] : e;

    return {
        x: (coords.pageX || coords.clientX + document.body.scrollLeft) - element.offsetLeft,
        y: (coords.pageY || coords.clientY + document.body.scrollTop) - element.offsetTop
    };
};