/// `fingertips` tworzy indeks odwrócony zbioru plikuów tekstowych.
///
/// Większość faktycznych operacji jest wykonywanya w modułach: `index`, 
/// `read`, `write` oraz `merge`. W tym pliku, `main.rs`, łączymy wszystkie 
/// te elementy w jedną całość na dwa różne sposoby.
///
/// *   Funkcja `run_single_threaded` wykonuje wszystkie operacje w jednym wątku,
///     w najprostszy możliwy sposób.
///
/// *   Następnie dzielimy cały proces na pięcioetapowy potok, dzięki czemu
///     możemy wykonywać operacje na większej liczbie procesorów. Funkcja
///     `run_pipeline` łączy te pięć etapów wjedną całość.
///
/// Funkcja `main` umieszczona na końcu pliku obsługuje parametry wiersza poleceń.
/// Następnie wywołuje jedną z dwóch powyższych funkcji w celu wykonania faktycznej
/// pracy.
/// 
mod index;
mod read;
mod write;
mod merge;
mod tmp;

use std::fs::File;
use std::io;
use std::io::prelude::*;
use std::path::{Path, PathBuf};
use std::sync::mpsc::{channel, Receiver};
use std::thread::{spawn, JoinHandle};
use argparse::{ArgumentParser, StoreTrue, Collect};

use crate::index::InMemoryIndex;
use crate::write::write_index_to_tmp_file;
use crate::merge::FileMerge;
use crate::tmp::TmpDir;

/// Tworzy indeks odwrócony podanej listy dokumentów i zapisuje go 
/// w podanym katalogu `output_dir`
fn run_single_threaded(documents: Vec<PathBuf>, output_dir: PathBuf)
    -> io::Result<()>
{
    // Jeśli wszystkie dokumenty można wygodnie wczytać do pamięci, to cały indeks
    // utworzymy w pamięci
    let mut accumulated_index = InMemoryIndex::new();

    // W przeciwnym razie, wraz z wypełnianiem pamięci, będziemy zapisywać na dysku
    // duże tymczasowe indeksy, zapisując nazwy plików tymaczsowych w `merge`, 
    // dzięki czemu później będziemy mogli je scalić w jeden wielki plik
    let mut merge = FileMerge::new(&output_dir);

    // Narzędzie do generacji nazw plików tymczasowych
    let mut tmp_dir = TmpDir::new(&output_dir);

    // Dla każdego dokumentu w zbiorze...
    for (doc_id, filename) in documents.into_iter().enumerate() {
        // ...wczytujemy go do pamięci...
        let mut f = File::open(filename)?;
        let mut text = String::new();
        f.read_to_string(&mut text)?;

        // ...i dodajemy jego zawartość do `accumulated_index` (przechowywanego w pamięci).
        let index = InMemoryIndex::from_single_document(doc_id, text);
        accumulated_index.merge(index);
        if accumulated_index.is_large() {
            // Aby uniknąć zapełnienia pamięci, zapisujemy `accumulated_index` na dysku
            let file = write_index_to_tmp_file(accumulated_index, &mut tmp_dir)?;
            merge.add_file(file)?;
            accumulated_index = InMemoryIndex::new();
        }
    }

    // Done reading documents! Save the last data set to disk, then merge the
    // temporary index files if there are more than one.
    if !accumulated_index.is_empty() {
        let file = write_index_to_tmp_file(accumulated_index, &mut tmp_dir)?;
        merge.add_file(file)?;
    }
    merge.finish()
}

/// Uruchamia wątek, który wczytuje do pamięci dokumenty z systemu plików.
///
/// `documents` to lista nazw plików do wczytania.
///
/// Funkcja zwraca parę wartości: obiekt `Receiver` pobierający dokumenty oraz 
/// `JoinHandle` którego można użyć do zaczekania na zakończenie tego wątku
/// i pobranie wartości `io::Error` w razie wystąpienia jakichś problemów.
fn start_file_reader_thread(documents: Vec<PathBuf>)
    -> (Receiver<String>, JoinHandle<io::Result<()>>)
{
    let (sender, receiver) = channel();

    let handle = spawn(move || {
        for filename in documents {
            let mut f = File::open(filename)?;
            let mut text = String::new();
            f.read_to_string(&mut text)?;

            if sender.send(text).is_err() {
                break;
            }
        }
        Ok(())
    });

    (receiver, handle)
}

/// Uruchamia wątek dzielący każdy dokument tekstowy na tokeny i konwertujący 
/// je na przechowywany w pamięci indeks. (Zakładamy, że każdy dokument można 
/// wczytać do pamięci.)
///
/// `texts` to strumień dokumentów zwrócony przez wątek odczytujący pliki.
///
/// Funkcja przypisuje każdemu dokumentowi numer. Zwraca parę wartości: obiekt
/// `Receiver` - sekwencję indeksów zapisanych w pamięcie; orz `JoinHandle` którego
/// można użyć do zaczekania na zakończenie tego wątku. Ten etap potoku przetwarzania
/// jest niezawodny (nie wykonuje żadnych operacji wejścia/wyjścia, więc błędy
/// nie mają gdzie się pojawiać)
fn start_file_indexing_thread(texts: Receiver<String>)
    -> (Receiver<InMemoryIndex>, JoinHandle<()>)
{
    let (sender, receiver) = channel();

    let handle = spawn(move || {
        for (doc_id, text) in texts.into_iter().enumerate() {
            let index = InMemoryIndex::from_single_document(doc_id, text);
            if sender.send(index).is_err() {
                break;
            }
        }
    });

    (receiver, handle)
}

/// Uruchamia wątek, który scala indeksy przechowywane w pamięci.
///
/// `file_indexes` to strumień indeksów z wątku indeksującego pliki.
/// Indeksy te zazwyczaj mają bardzo różne wielkości, gdyż także dokumenty wejściowe
/// będą miały różną wielkość.
///
/// Wątek utworzony przez tę funkcję scala wszystkie te indeksy w "duże"
/// indeksy i przekazuje te duże indeksy do nowego kanału.
///
/// Funkcja zwraca parę: obiekt `Receiver` - sekwencję dużych indeksów utworzonych
/// poprzez scalenie indeksów wejściowych oraz `JoinHandle` który można użyć by
/// zaczekać na zakończenie tego wątku. Ten etap potoku przetwarzania
/// jest niezawodny (nie wykonuje żadnych operacji wejścia/wyjścia, więc błędy
/// nie mają gdzie się pojawiać)
fn start_in_memory_merge_thread(file_indexes: Receiver<InMemoryIndex>)
    -> (Receiver<InMemoryIndex>, JoinHandle<()>)
{
    let (sender, receiver) = channel();

    let handle = spawn(move || {
        let mut accumulated_index = InMemoryIndex::new();
        for fi in file_indexes {
            accumulated_index.merge(fi);
            if accumulated_index.is_large() {
                if sender.send(accumulated_index).is_err() {
                    return;
                }
                accumulated_index = InMemoryIndex::new();
            }
        }
        if !accumulated_index.is_empty() {
            let _ = sender.send(accumulated_index);
        }
    });

    (receiver, handle)
}

/// Uruchamia wątek, który zapisuje duże indeksy do plików tymczasowych.
///
/// Ten wątek, dla każdego indeksu w `big_indexes`, generuje plik o pozbawionej
/// znaczenia unikalnej nazwie, zapisuje w nim dane i przekazuje nazwę pliku
/// do nowego kanału.
///
/// Funkcja zwraca parę: obiekt `Receiver` odbierający nazwy plików oraz 
/// `JoinHandle` który można użyć by zaczekać na zakończenie tego wątku i pobrać
/// ewentualne błędy wejścia/wyjścia
fn start_index_writer_thread(big_indexes: Receiver<InMemoryIndex>,
                             output_dir: &Path)
    -> (Receiver<PathBuf>, JoinHandle<io::Result<()>>)
{
    let (sender, receiver) = channel();

    let mut tmp_dir = TmpDir::new(output_dir);
    let handle = spawn(move || {
        for index in big_indexes {
            let file = write_index_to_tmp_file(index, &mut tmp_dir)?;
            if sender.send(file).is_err() {
                break;
            }
        }
        Ok(())
    });

    (receiver, handle)
}

/// Ta funkcja, na podstawie sekwencji nazw plików z danymi indeksu, scala zawartość
/// tych plików i zapisuje w jednym dużym pliku indeksu
fn merge_index_files(files: Receiver<PathBuf>, output_dir: &Path)
    -> io::Result<()>
{
    let mut merge = FileMerge::new(output_dir);
    for file in files {
        merge.add_file(file)?;
    }
    merge.finish()
}

/// Tworzy indeks odwrócony przekazanej listy dokumentów `documents` i zapisuje
/// go w `output_dir`.
///
/// W razie powodzienia, funkcja robi to samo co `run_single_threaded`, choć robi
/// to szybciej, gdyż używa wielu rdzeni procesora i zapewnia ich działanie 
/// podczas wykonywania operacji wejścia/wyjścia
fn run_pipeline(documents: Vec<PathBuf>, output_dir: PathBuf)
    -> io::Result<()>
{
    // Uruchamiamy wszystkie pięć etapów potoku
    let (texts,   h1) = start_file_reader_thread(documents);
    let (pints,   h2) = start_file_indexing_thread(texts);
    let (gallons, h3) = start_in_memory_merge_thread(pints);
    let (files,   h4) = start_index_writer_thread(gallons, &output_dir);
    let result = merge_index_files(files, &output_dir);

    // Oczekujemy na zakończenie wątków, zatrzymując działanie w razie napotkania błędów
    let r1 = h1.join().unwrap();
    h2.join().unwrap();
    h3.join().unwrap();
    let r4 = h4.join().unwrap();

    // Zwracamy pierwszy napotkany błąd (jeśli się pojawił).
    // (Wiadomo, że h2 i h3 nie mogą zawieść - w tych wątkach przetwarzanie
    // jest wykonywane w całości w pamięci.)
    r1?;
    r4?;
    result
}

/// Dysponujące pewnymi ścieżkami, funkcja generuje kompletną listę plików tekstowych
/// do zindeksowania. Sprawdzamy czy ścieśka odnosi się do katalogu czy pliku;
/// w przypadku katalogu do zindeksowania zostaną automatycznie wybrane wszystkie 
/// pliki .txt zapisane w tym katalogu. Uwzględniane są także ścieżki względne
///
/// Jeśli któraś ze ścieżek w `args` nie będzie prawidłową ścieżką do pliku lub 
/// katalogu, zostanie to potraktowane jako błąd
fn expand_filename_arguments(args: Vec<String>) -> io::Result<Vec<PathBuf>> {
    let mut filenames = vec![];
    for arg in args {
        let path = PathBuf::from(arg);
        if path.metadata()?.is_dir() {
            for entry in path.read_dir()? {
                let entry = entry?;
                if entry.file_type()?.is_file() {
                    filenames.push(entry.path());
                }
            }
        } else {
            filenames.push(path);
        }
    }
    Ok(filenames)
}

/// Generuje indeks dla grupy plików tekstowych
fn run(filenames: Vec<String>, single_threaded: bool) -> io::Result<()> {
    let output_dir = PathBuf::from(".");
    let documents = expand_filename_arguments(filenames)?;

    if single_threaded {
        run_single_threaded(documents, output_dir)
    } else {
        run_pipeline(documents, output_dir)
    }
}

fn main() {
    let mut single_threaded = false;
    let mut filenames = vec![];

    {
        let mut ap = ArgumentParser::new();
        ap.set_description("Tworzy indeks odwrócony do przeszukiwania dokumentów tekstowych.");
        ap.refer(&mut single_threaded)
            .add_option(&["-1", "--single-threaded"], StoreTrue,
                        "Wszystkie operacje zostaną wykonane w jednym wątku.");
        ap.refer(&mut filenames)
            .add_argument("filenames", Collect,
                          "Nazwy plików lub katalogów do zindeksowania. \
                           W przypadku katalogów, automatycznie zostaną zindeksowane \
                           wszystkie pliki .txt umieszczone w tym katalogu.");
        ap.parse_args_or_exit();
    }

    match run(filenames, single_threaded) {
        Ok(()) => {}
        Err(err) => println!("błąd: {}", err)
    }
}
